package datamaxoneil.printer.configuration;

/**
 * This class handles the data from an {TC?} query as well as
 * accessors to read the data values.
 * <p>
 * This class supports the following possible parameters:
 * <table border=1>
 *    <tr><th>Parameter</th><th>Description</th></tr>
 *    <tr><td>802.11B Info</td><td>Wireless Card Info</td></tr>
 *    <tr><td>C</td><td>Valid Certificate Present</td></tr>
 *    <tr><td>CR</td><td>When enabled this will cause the printer to report when the TCP connection is ready, or the number of connections ahead of them.</td></tr>
 *    <tr><td>D</td><td>Acquire IP Settings</td></tr>
 *    <tr><td>DR</td><td>Is the 802.11 radio disabled.</td></tr>
 *    <tr><td>E</td><td>ESSID</td></tr>
 *    <tr><td>ET</td><td>EAP Type</td></tr>
 *    <tr><td>G</td><td>Printers Gateway Address</td></tr>
 *    <tr><td>I</td><td>Printers IP Address</td></tr>
 *    <tr><td>I2</td><td>Inactivity Timeout</td></tr>
 *    <tr><td>K</td><td>Key To Use</td></tr>
 *    <tr><td>K1</td><td>Key1</td></tr>
 *    <tr><td>K2</td><td>Key2</td></tr>
 *    <tr><td>K3</td><td>Key3</td></tr>
 *    <tr><td>K4</td><td>Key4</td></tr>
 *    <tr><td>M</td><td>Printers Subnet Mask</td></tr>
 *    <tr><td>MAC</td><td>MAC Address</td></tr>
 *    <tr><td>MX</td><td>Migration Mode Enabled</td></tr>
 *    <tr><td>N</td><td>Station Name</td></tr>
 *    <tr><td>NA</td><td>Network Authentication</td></tr>
 *    <tr><td>P</td><td>TCP Printing Port</td></tr>
 *    <tr><td>P2</td><td>Power Saving Mode</td></tr>
 *    <tr><td>P2M</td><td>Phase 2 Method</td></tr>
 *    <tr><td>P3</td><td>UDP Printing Port</td></tr>
 *    <tr><td>PWR</td><td>Card Powered</td></tr>
 *    <tr><td>Q</td><td>Signal Quality Indicator</td></tr>
 *    <tr><td>S</td><td>Authentication Algorithm</td></tr>
 *    <tr><td>T</td><td>Station Type</td></tr>
 *    <tr><td>W</td><td>Encryption Enabled</td></tr>
 *    <tr><td>Z</td><td>Current Certificate CRC</td></tr>
 * </table>
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class TCPIPStatus extends PrinterState {

	/**
	 * Acquire IP Settings
	 */
	public enum IPAquiringValues {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * Use BOOTP to acquire an IP address.
		 */
		BOOTP(66),

		/**
		 * Failed to acquire an IP address.
		 */
		RequestFailed(70),

		/**
		 * Use a statically defined IP address.
		 */
		StaticIP(78),

		/**
		 * Currently requesting an IP address.
		 */
		RequestingIP(82),

		/**
		 * Use DHCP to acquire an IP address.
		 */
		DHCP(89);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		IPAquiringValues(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * EAP Type
	 */
	public enum EAPTypeValues {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * Generic Token Card
		 */
		NONE(0),

		/**
		 * Identity Request
		 */
		IDENTITY(1),

		/**
		 * Transport Level Security
		 */
		NOTIFICATION(2),

		/**
		 * One Time Password
		 */
		NAK(3),

		/**
		 * NAK
		 */
		MD5(4),

		/**
		 * Lightweight Extensible Authentication Protocol
		 */
		OTP(5),

		/**
		 * None, are you feeling lucky?
		 */
		GTC(6),

		/**
		 * Protected Transport Level Security
		 */
		TLS(13),

		/**
		 * Notification
		 */
		LEAP(17),

		/**
		 * Tunneled Transport Level Security
		 */
		SIM(18),

		/**
		 * Type-length-value
		 */
		TTLS(21),

		/**
		 * GSM Subscriber Identity 
		 */
		PEAP(25),

		/**
		 * Message Digest 5
		 */
		MSCHAPV2(26),

		/**
		 * MSCHAPV2 version of PEAP
		 */
		TLV(33),

		/**
		 * The result was either unrecognized or never set.
		 */
		FAST(43);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		EAPTypeValues(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Key1
	 */
	public enum AuthenticationKeyTypeValues {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * No key present
		 */
		None(0),

		/**
		 * 64 bits
		 */
		PSK_64Bits(5),

		/**
		 * 128 bits
		 */
		PSK_128Bits(13);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		AuthenticationKeyTypeValues(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Network Authentication
	 */
	public enum NetworkAuthenticationValues {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * No Encryption
		 */
		None(0),

		/**
		 * Lightweight Extensible Authentication Protocol
		 */
		LEAP(1),

		/**
		 * Wi-Fi Protected Access Version 1 with pre-shared key
		 */
		WPA_PSK(2),

		/**
		 * Wi-Fi Protected Access Version 1
		 */
		WPA(3),

		/**
		 * Wi-Fi Protected Access Version 1 with Lightweight Extensible Authentication Protocol
		 */
		WPA_LEAP(4),

		/**
		 * Wi-Fi Protected Access Version 2 with pre-shared key
		 */
		WPA2_PSK(5),

		/**
		 * Wi-Fi Protected Access Version 2
		 */
		WPA2(6),

		/**
		 * For use when another host based EAP is used.
		 */
		HostBasedEAP(7);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		NetworkAuthenticationValues(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Phase 2 Method
	 */
	public enum Phase2MethodValues {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * Use EAP_GTC
		 */
		EAP_GTC(1),

		/**
		 * Use EAP_MSCHAPV2
		 */
		EAP_MSCHAPV2(2),

		/**
		 * Use EAP_MD5
		 */
		EAP_MD5(3),

		/**
		 * Use GTC
		 */
		GTC(4),

		/**
		 * Use MSCHAPV2
		 */
		MSCHAPV2(5),

		/**
		 * Use MSCHAP
		 */
		MSCHAP(6),

		/**
		 * Use CHAP
		 */
		CHAP(7),

		/**
		 * Use EAP_TLS
		 */
		EAP_TLS(8);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		Phase2MethodValues(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Station Type
	 */
	public enum NetworkTypeValues {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * Currently configured for an AdHoc network.
		 */
		AD_HOC(72),

		/**
		 * Currently configured to use an access point.
		 */
		AccessPoint(80);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		NetworkTypeValues(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * The default constructor will initialize the class with default values.
	 * <p>
	 * Initially all of the _IsValid parameters will be false because no data has been processed.  To 'populate' the values, the object must invoke <see cref="PrinterState.Update(ONeil.Connection.ConnectionBase, int)">Update</see> with the query response string.
	 */
	public TCPIPStatus() {

		// Set Query Values
		m_QueryDescription = "TCPIP Status";
		m_Query = "{TC?}";
		m_QueryResponseHeader = "{TC!";

		// Add names
		addName("802.11B Info", "Wireless Card Info");
		addName("C", "Valid Certificate Present");
		addName("CR", "Connection Reporting Enabled");
		addName("D", "Acquire IP");
		addName("DR", "Wireless Radio State");
		addName("E", "ESSID");
		addName("ET", "EAP Type");
		addName("G", "Gateway Address");
		addName("I", "IP Address");
		addName("I2", "Inactivity Timeout");
		addName("K", "Key To Use");
		addName("K1", "Key1");
		addName("K2", "Key2");
		addName("K3", "Key3");
		addName("K4", "Key4");
		addName("M", "Subnet Mask");
		addName("MAC", "MAC Address");
		addName("MX", "Migration Mode Enabled");
		addName("N", "Station Name");
		addName("NA", "Network Authentication");
		addName("P", "TCP Printing Port");
		addName("P2", "Power Saving Mode");
		addName("P2M", "Phase 2 Method");
		addName("P3", "UDP Printing Port");
		addName("PWR", "Card Powered");
		addName("Q", "Signal Quality Indicator");
		addName("S", "Authentication Algorithm");
		addName("T", "Station Type");
		addName("W", "Encryption Enabled");
		addName("Z", "Current Certificate CRC");

		return;
	}
	/**
	 * Indicates if getWirelessCardInfo() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getWirelessCardInfo() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getWirelessCardInfo()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getWirelessCardInfo_IsPresent() {
		return containsData("802.11B Info") && isString("802.11B Info");
	}

	/**
	 * Wireless Card Info
	 * <p>
	 * The return value for this parameter is only valid if getWirelessCardInfo_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getWirelessCardInfo_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getWirelessCardInfo() {
		return parse_string("802.11B Info");
	}

	/**
	 * Indicates if getValidCertificatePresent() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getValidCertificatePresent() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getValidCertificatePresent()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getValidCertificatePresent_IsPresent() {
		return containsData("C") && isString("C");
	}

	/**
	 * Valid Certificate Present
	 * <p>
	 * The return value for this parameter is only valid if getValidCertificatePresent_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getValidCertificatePresent_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getValidCertificatePresent() {
		return parse_boolean("C", "Y", "N");
	}

	/**
	 * Indicates if getConnectionReporting() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getConnectionReporting() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getConnectionReporting()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getConnectionReporting_IsPresent() {
		return containsData("CR") && isString("CR");
	}

	/**
	 * When enabled this will cause the printer to report when the TCP connection is ready, or the number of connections ahead of them.
	 * <p>
	 * The return value for this parameter is only valid if getConnectionReporting_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getConnectionReporting_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getConnectionReporting() {
		return parse_boolean("CR", "Y", "N");
	}

	/**
	 * Indicates if getAcquireIP() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getAcquireIP() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getAcquireIP()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getAcquireIP_IsPresent() {
		return containsData("D") && isString("D");
	}

	/**
	 * Acquire IP Settings
	 * <p>
	 * The return value for this parameter is only valid if getAcquireIP_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getAcquireIP_IsPresent()
	 * @return The current value for the parameter.
	 */
	public IPAquiringValues getAcquireIP() {
		IPAquiringValues result = IPAquiringValues.Unset;

		// Determine value
		if (!containsData("D")) {
			// Value not set
		}
		else if (queryResult("D").equals("B")) {
			// Use BOOTP to acquire an IP address.
			result = IPAquiringValues.BOOTP;
		}
		else if (queryResult("D").equals("F")) {
			// Failed to acquire an IP address.
			result = IPAquiringValues.RequestFailed;
		}
		else if (queryResult("D").equals("N")) {
			// Use a statically defined IP address.
			result = IPAquiringValues.StaticIP;
		}
		else if (queryResult("D").equals("R")) {
			// Currently requesting an IP address.
			result = IPAquiringValues.RequestingIP;
		}
		else if (queryResult("D").equals("Y")) {
			// Use DHCP to acquire an IP address.
			result = IPAquiringValues.DHCP;
		}
		else {
			// Value not yet in DB
			result = IPAquiringValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getRadioDisabled() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRadioDisabled() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getRadioDisabled()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRadioDisabled_IsPresent() {
		return containsData("DR") && isString("DR");
	}

	/**
	 * Is the 802.11 radio disabled.
	 * <p>
	 * The return value for this parameter is only valid if getRadioDisabled_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getRadioDisabled_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getRadioDisabled() {
		return parse_boolean("DR", "1", "0");
	}

	/**
	 * Indicates if getESSID() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getESSID() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getESSID()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getESSID_IsPresent() {
		return containsData("E") && isString("E");
	}

	/**
	 * ESSID
	 * <p>
	 * The return value for this parameter is only valid if getESSID_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getESSID_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getESSID() {
		return parse_string("E");
	}

	/**
	 * Indicates if getEAPType() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getEAPType() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getEAPType()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getEAPType_IsPresent() {
		return containsData("ET") && isString("ET");
	}

	/**
	 * EAP Type
	 * <p>
	 * The return value for this parameter is only valid if getEAPType_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getEAPType_IsPresent()
	 * @return The current value for the parameter.
	 */
	public EAPTypeValues getEAPType() {
		EAPTypeValues result = EAPTypeValues.Unset;

		// Determine value
		if (!containsData("ET")) {
			// Value not set
		}
		else if (queryResult("ET").equals("0")) {
			// Generic Token Card
			result = EAPTypeValues.NONE;
		}
		else if (queryResult("ET").equals("1")) {
			// Identity Request
			result = EAPTypeValues.IDENTITY;
		}
		else if (queryResult("ET").equals("2")) {
			// Transport Level Security
			result = EAPTypeValues.NOTIFICATION;
		}
		else if (queryResult("ET").equals("3")) {
			// One Time Password
			result = EAPTypeValues.NAK;
		}
		else if (queryResult("ET").equals("4")) {
			// NAK
			result = EAPTypeValues.MD5;
		}
		else if (queryResult("ET").equals("5")) {
			// Lightweight Extensible Authentication Protocol
			result = EAPTypeValues.OTP;
		}
		else if (queryResult("ET").equals("6")) {
			// None, are you feeling lucky?
			result = EAPTypeValues.GTC;
		}
		else if (queryResult("ET").equals("13")) {
			// Protected Transport Level Security
			result = EAPTypeValues.TLS;
		}
		else if (queryResult("ET").equals("17")) {
			// Notification
			result = EAPTypeValues.LEAP;
		}
		else if (queryResult("ET").equals("18")) {
			// Tunneled Transport Level Security
			result = EAPTypeValues.SIM;
		}
		else if (queryResult("ET").equals("21")) {
			// Type-length-value
			result = EAPTypeValues.TTLS;
		}
		else if (queryResult("ET").equals("25")) {
			// GSM Subscriber Identity 
			result = EAPTypeValues.PEAP;
		}
		else if (queryResult("ET").equals("26")) {
			// Message Digest 5
			result = EAPTypeValues.MSCHAPV2;
		}
		else if (queryResult("ET").equals("33")) {
			// MSCHAPV2 version of PEAP
			result = EAPTypeValues.TLV;
		}
		else if (queryResult("ET").equals("43")) {
			// The result was either unrecognized or never set.
			result = EAPTypeValues.FAST;
		}
		else {
			// Value not yet in DB
			result = EAPTypeValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getGatewayAddress() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getGatewayAddress() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getGatewayAddress()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getGatewayAddress_IsPresent() {
		return containsData("G") && isString("G");
	}

	/**
	 * Printers Gateway Address
	 * <p>
	 * The return value for this parameter is only valid if getGatewayAddress_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getGatewayAddress_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getGatewayAddress() {
		return parse_string("G");
	}

	/**
	 * Indicates if getIPAddress() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getIPAddress() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getIPAddress()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getIPAddress_IsPresent() {
		return containsData("I") && isString("I");
	}

	/**
	 * Printers IP Address
	 * <p>
	 * The return value for this parameter is only valid if getIPAddress_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getIPAddress_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getIPAddress() {
		return parse_string("I");
	}

	/**
	 * Indicates if getInactivityTimeout() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getInactivityTimeout() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getInactivityTimeout()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getInactivityTimeout_IsPresent() {
		return containsData("I2") && isInteger("I2");
	}

	/**
	 * Inactivity Timeout
	 * <p>
	 * The return value for this parameter is only valid if getInactivityTimeout_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getInactivityTimeout_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getInactivityTimeout() {
		return parse_long("I2");
	}

	/**
	 * Indicates if getKeyToUse() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getKeyToUse() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKeyToUse()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getKeyToUse_IsPresent() {
		return containsData("K") && isInteger("K");
	}

	/**
	 * Key To Use
	 * <p>
	 * The return value for this parameter is only valid if getKeyToUse_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKeyToUse_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getKeyToUse() {
		return parse_long("K");
	}

	/**
	 * Indicates if getKey1() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getKey1() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKey1()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getKey1_IsPresent() {
		return containsData("K1") && isString("K1");
	}

	/**
	 * Key1
	 * <p>
	 * The return value for this parameter is only valid if getKey1_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKey1_IsPresent()
	 * @return The current value for the parameter.
	 */
	public AuthenticationKeyTypeValues getKey1() {
		AuthenticationKeyTypeValues result = AuthenticationKeyTypeValues.Unset;

		// Determine value
		if (!containsData("K1")) {
			// Value not set
		}
		else if (queryResult("K1").equals("0")) {
			// No key present
			result = AuthenticationKeyTypeValues.None;
		}
		else if (queryResult("K1").equals("5")) {
			// 64 bits
			result = AuthenticationKeyTypeValues.PSK_64Bits;
		}
		else if (queryResult("K1").equals("13")) {
			// 128 bits
			result = AuthenticationKeyTypeValues.PSK_128Bits;
		}
		else {
			// Value not yet in DB
			result = AuthenticationKeyTypeValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getKey2() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getKey2() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKey2()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getKey2_IsPresent() {
		return containsData("K2") && isString("K2");
	}

	/**
	 * Key2
	 * <p>
	 * The return value for this parameter is only valid if getKey2_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKey2_IsPresent()
	 * @return The current value for the parameter.
	 */
	public AuthenticationKeyTypeValues getKey2() {
		AuthenticationKeyTypeValues result = AuthenticationKeyTypeValues.Unset;

		// Determine value
		if (!containsData("K2")) {
			// Value not set
		}
		else if (queryResult("K2").equals("0")) {
			// No key present
			result = AuthenticationKeyTypeValues.None;
		}
		else if (queryResult("K2").equals("5")) {
			// 64 bits
			result = AuthenticationKeyTypeValues.PSK_64Bits;
		}
		else if (queryResult("K2").equals("13")) {
			// 128 bits
			result = AuthenticationKeyTypeValues.PSK_128Bits;
		}
		else {
			// Value not yet in DB
			result = AuthenticationKeyTypeValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getKey3() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getKey3() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKey3()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getKey3_IsPresent() {
		return containsData("K3") && isString("K3");
	}

	/**
	 * Key3
	 * <p>
	 * The return value for this parameter is only valid if getKey3_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKey3_IsPresent()
	 * @return The current value for the parameter.
	 */
	public AuthenticationKeyTypeValues getKey3() {
		AuthenticationKeyTypeValues result = AuthenticationKeyTypeValues.Unset;

		// Determine value
		if (!containsData("K3")) {
			// Value not set
		}
		else if (queryResult("K3").equals("0")) {
			// No key present
			result = AuthenticationKeyTypeValues.None;
		}
		else if (queryResult("K3").equals("5")) {
			// 64 bits
			result = AuthenticationKeyTypeValues.PSK_64Bits;
		}
		else if (queryResult("K3").equals("13")) {
			// 128 bits
			result = AuthenticationKeyTypeValues.PSK_128Bits;
		}
		else {
			// Value not yet in DB
			result = AuthenticationKeyTypeValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getKey4() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getKey4() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKey4()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getKey4_IsPresent() {
		return containsData("K4") && isString("K4");
	}

	/**
	 * Key4
	 * <p>
	 * The return value for this parameter is only valid if getKey4_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getKey4_IsPresent()
	 * @return The current value for the parameter.
	 */
	public AuthenticationKeyTypeValues getKey4() {
		AuthenticationKeyTypeValues result = AuthenticationKeyTypeValues.Unset;

		// Determine value
		if (!containsData("K4")) {
			// Value not set
		}
		else if (queryResult("K4").equals("0")) {
			// No key present
			result = AuthenticationKeyTypeValues.None;
		}
		else if (queryResult("K4").equals("5")) {
			// 64 bits
			result = AuthenticationKeyTypeValues.PSK_64Bits;
		}
		else if (queryResult("K4").equals("13")) {
			// 128 bits
			result = AuthenticationKeyTypeValues.PSK_128Bits;
		}
		else {
			// Value not yet in DB
			result = AuthenticationKeyTypeValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getSubnetMask() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getSubnetMask() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getSubnetMask()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getSubnetMask_IsPresent() {
		return containsData("M") && isString("M");
	}

	/**
	 * Printers Subnet Mask
	 * <p>
	 * The return value for this parameter is only valid if getSubnetMask_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getSubnetMask_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getSubnetMask() {
		return parse_string("M");
	}

	/**
	 * Indicates if getMACAddress() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getMACAddress() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getMACAddress()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getMACAddress_IsPresent() {
		return containsData("MAC") && isString("MAC");
	}

	/**
	 * MAC Address
	 * <p>
	 * The return value for this parameter is only valid if getMACAddress_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getMACAddress_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getMACAddress() {
		return parse_string("MAC");
	}

	/**
	 * Indicates if getMigrationModeEnabled() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getMigrationModeEnabled() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getMigrationModeEnabled()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getMigrationModeEnabled_IsPresent() {
		return containsData("MX") && isString("MX");
	}

	/**
	 * Migration Mode Enabled
	 * <p>
	 * The return value for this parameter is only valid if getMigrationModeEnabled_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getMigrationModeEnabled_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getMigrationModeEnabled() {
		return parse_boolean("MX", "1", "0");
	}

	/**
	 * Indicates if getStationName() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getStationName() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getStationName()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getStationName_IsPresent() {
		return containsData("N") && isString("N");
	}

	/**
	 * Station Name
	 * <p>
	 * The return value for this parameter is only valid if getStationName_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getStationName_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getStationName() {
		return parse_string("N");
	}

	/**
	 * Indicates if getNetworkAuthentication() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getNetworkAuthentication() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getNetworkAuthentication()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getNetworkAuthentication_IsPresent() {
		return containsData("NA") && isString("NA");
	}

	/**
	 * Network Authentication
	 * <p>
	 * The return value for this parameter is only valid if getNetworkAuthentication_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getNetworkAuthentication_IsPresent()
	 * @return The current value for the parameter.
	 */
	public NetworkAuthenticationValues getNetworkAuthentication() {
		NetworkAuthenticationValues result = NetworkAuthenticationValues.Unset;

		// Determine value
		if (!containsData("NA")) {
			// Value not set
		}
		else if (queryResult("NA").equals("0")) {
			// No Encryption
			result = NetworkAuthenticationValues.None;
		}
		else if (queryResult("NA").equals("1")) {
			// Lightweight Extensible Authentication Protocol
			result = NetworkAuthenticationValues.LEAP;
		}
		else if (queryResult("NA").equals("2")) {
			// Wi-Fi Protected Access Version 1 with pre-shared key
			result = NetworkAuthenticationValues.WPA_PSK;
		}
		else if (queryResult("NA").equals("3")) {
			// Wi-Fi Protected Access Version 1
			result = NetworkAuthenticationValues.WPA;
		}
		else if (queryResult("NA").equals("4")) {
			// Wi-Fi Protected Access Version 1 with Lightweight Extensible Authentication Protocol
			result = NetworkAuthenticationValues.WPA_LEAP;
		}
		else if (queryResult("NA").equals("5")) {
			// Wi-Fi Protected Access Version 2 with pre-shared key
			result = NetworkAuthenticationValues.WPA2_PSK;
		}
		else if (queryResult("NA").equals("6")) {
			// Wi-Fi Protected Access Version 2
			result = NetworkAuthenticationValues.WPA2;
		}
		else if (queryResult("NA").equals("7")) {
			// For use when another host based EAP is used.
			result = NetworkAuthenticationValues.HostBasedEAP;
		}
		else {
			// Value not yet in DB
			result = NetworkAuthenticationValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getTCPPrintingPort() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getTCPPrintingPort() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getTCPPrintingPort()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getTCPPrintingPort_IsPresent() {
		return containsData("P") && isInteger("P");
	}

	/**
	 * TCP Printing Port
	 * <p>
	 * The return value for this parameter is only valid if getTCPPrintingPort_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getTCPPrintingPort_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getTCPPrintingPort() {
		return parse_long("P");
	}

	/**
	 * Indicates if getPowerSavingMode() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPowerSavingMode() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getPowerSavingMode()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPowerSavingMode_IsPresent() {
		return containsData("P2") && isString("P2");
	}

	/**
	 * Power Saving Mode
	 * <p>
	 * The return value for this parameter is only valid if getPowerSavingMode_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getPowerSavingMode_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getPowerSavingMode() {
		return parse_boolean("P2", "Y", "N");
	}

	/**
	 * Indicates if getPhase2Method() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPhase2Method() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getPhase2Method()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPhase2Method_IsPresent() {
		return containsData("P2M") && isString("P2M");
	}

	/**
	 * Phase 2 Method
	 * <p>
	 * The return value for this parameter is only valid if getPhase2Method_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getPhase2Method_IsPresent()
	 * @return The current value for the parameter.
	 */
	public Phase2MethodValues getPhase2Method() {
		Phase2MethodValues result = Phase2MethodValues.Unset;

		// Determine value
		if (!containsData("P2M")) {
			// Value not set
		}
		else if (queryResult("P2M").equals("1")) {
			// Use EAP_GTC
			result = Phase2MethodValues.EAP_GTC;
		}
		else if (queryResult("P2M").equals("2")) {
			// Use EAP_MSCHAPV2
			result = Phase2MethodValues.EAP_MSCHAPV2;
		}
		else if (queryResult("P2M").equals("3")) {
			// Use EAP_MD5
			result = Phase2MethodValues.EAP_MD5;
		}
		else if (queryResult("P2M").equals("4")) {
			// Use GTC
			result = Phase2MethodValues.GTC;
		}
		else if (queryResult("P2M").equals("5")) {
			// Use MSCHAPV2
			result = Phase2MethodValues.MSCHAPV2;
		}
		else if (queryResult("P2M").equals("6")) {
			// Use MSCHAP
			result = Phase2MethodValues.MSCHAP;
		}
		else if (queryResult("P2M").equals("7")) {
			// Use CHAP
			result = Phase2MethodValues.CHAP;
		}
		else if (queryResult("P2M").equals("8")) {
			// Use EAP_TLS
			result = Phase2MethodValues.EAP_TLS;
		}
		else {
			// Value not yet in DB
			result = Phase2MethodValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getUDPPrintingPort() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getUDPPrintingPort() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getUDPPrintingPort()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getUDPPrintingPort_IsPresent() {
		return containsData("P3") && isInteger("P3");
	}

	/**
	 * UDP Printing Port
	 * <p>
	 * The return value for this parameter is only valid if getUDPPrintingPort_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getUDPPrintingPort_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getUDPPrintingPort() {
		return parse_long("P3");
	}

	/**
	 * Indicates if getCardPowered() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getCardPowered() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getCardPowered()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getCardPowered_IsPresent() {
		return containsData("PWR") && isString("PWR");
	}

	/**
	 * Card Powered
	 * <p>
	 * The return value for this parameter is only valid if getCardPowered_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getCardPowered_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getCardPowered() {
		return parse_boolean("PWR", "ON", "OFF");
	}

	/**
	 * Indicates if getSignalQualityIndicator() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getSignalQualityIndicator() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getSignalQualityIndicator()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getSignalQualityIndicator_IsPresent() {
		return containsData("Q") && isString("Q");
	}

	/**
	 * Signal Quality Indicator
	 * <p>
	 * The return value for this parameter is only valid if getSignalQualityIndicator_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getSignalQualityIndicator_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getSignalQualityIndicator() {
		return parse_boolean("Q", "Y", "N");
	}

	/**
	 * Indicates if getAuthenticationAlgorithm() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getAuthenticationAlgorithm() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getAuthenticationAlgorithm()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getAuthenticationAlgorithm_IsPresent() {
		return containsData("S") && isString("S");
	}

	/**
	 * Authentication Algorithm
	 * <p>
	 * The return value for this parameter is only valid if getAuthenticationAlgorithm_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getAuthenticationAlgorithm_IsPresent()
	 * @return The current value for the parameter.
	 */
	public AuthenticationKeyTypeValues getAuthenticationAlgorithm() {
		AuthenticationKeyTypeValues result = AuthenticationKeyTypeValues.Unset;

		// Determine value
		if (!containsData("S")) {
			// Value not set
		}
		else if (queryResult("S").equals("0")) {
			// No key present
			result = AuthenticationKeyTypeValues.None;
		}
		else if (queryResult("S").equals("5")) {
			// 64 bits
			result = AuthenticationKeyTypeValues.PSK_64Bits;
		}
		else if (queryResult("S").equals("13")) {
			// 128 bits
			result = AuthenticationKeyTypeValues.PSK_128Bits;
		}
		else {
			// Value not yet in DB
			result = AuthenticationKeyTypeValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getStationType() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getStationType() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getStationType()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getStationType_IsPresent() {
		return containsData("T") && isString("T");
	}

	/**
	 * Station Type
	 * <p>
	 * The return value for this parameter is only valid if getStationType_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getStationType_IsPresent()
	 * @return The current value for the parameter.
	 */
	public NetworkTypeValues getStationType() {
		NetworkTypeValues result = NetworkTypeValues.Unset;

		// Determine value
		if (!containsData("T")) {
			// Value not set
		}
		else if (queryResult("T").equals("H")) {
			// Currently configured for an AdHoc network.
			result = NetworkTypeValues.AD_HOC;
		}
		else if (queryResult("T").equals("P")) {
			// Currently configured to use an access point.
			result = NetworkTypeValues.AccessPoint;
		}
		else {
			// Value not yet in DB
			result = NetworkTypeValues.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getEncryptionEnabled() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getEncryptionEnabled() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getEncryptionEnabled()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getEncryptionEnabled_IsPresent() {
		return containsData("W") && isInteger("W");
	}

	/**
	 * Encryption Enabled
	 * <p>
	 * The return value for this parameter is only valid if getEncryptionEnabled_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getEncryptionEnabled_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getEncryptionEnabled() {
		return parse_long("W");
	}

	/**
	 * Indicates if getCurrentCertificateCRC() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getCurrentCertificateCRC() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getCurrentCertificateCRC()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getCurrentCertificateCRC_IsPresent() {
		return containsData("Z") && isString("Z");
	}

	/**
	 * Current Certificate CRC
	 * <p>
	 * The return value for this parameter is only valid if getCurrentCertificateCRC_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see TCPIPStatus#getCurrentCertificateCRC_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getCurrentCertificateCRC() {
		return parse_string("Z");
	}

}
