package datamaxoneil.printer.configuration;

/**
 * This class handles the data from an {S2?} query as well as
 * accessors to read the data values.
 * <p>
 * This class supports the following possible parameters:
 * <table border=1>
 *    <tr><th>Parameter</th><th>Description</th></tr>
 *    <tr><td>B</td><td>Serial Baud Rate</td></tr>
 *    <tr><td>H</td><td>Handshake type</td></tr>
 *    <tr><td>N</td><td>Data Bits</td></tr>
 *    <tr><td>P</td><td>Serial Communication Parity</td></tr>
 * </table>
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class SSP_CommParameters extends PrinterState {

	/**
	 * Serial Baud Rate
	 */
	public enum BaudValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * 1200 Baud
		 */
		Baud_1200(1200),

		/**
		 * 2400 Baud
		 */
		Baud_2400(2400),

		/**
		 * 4800 Baud
		 */
		Baud_4800(4800),

		/**
		 * 9600 Baud
		 */
		Baud_9600(9600),

		/**
		 * 19200 Baud
		 */
		Baud_19200(19200),

		/**
		 * 38400 Baud
		 */
		Baud_38400(38400),

		/**
		 * 57600 Baud
		 */
		Baud_57600(57600),

		/**
		 * 115200 Baud
		 */
		Baud_115200(115200),

		/**
		 * 230400 Baud
		 */
		Baud_230400(230400),

		/**
		 * 460800 Baud
		 */
		Baud_460800(460800),

		/**
		 * 921600 Baud
		 */
		Baud_921600(921600);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		BaudValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Handshake type
	 */
	public enum Handshake {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * No control is used for the handshake
		 */
		None(0),

		/**
		 * Request-to-Send (RTS) hardware flow control is used. RTS signals that data is available for transmission. If the input buffer becomes full, the RTS line will be set to false. The RTS line will be set to true when more room becomes available in the input buffer
		 */
		Hardware(1),

		/**
		 * The XON/XOFF software control protocol is used. The XOFF control is sent to stop the transmission of data. The XON control is sent to resume the transmission. These software controls are used instead of Request to Send (RTS) and Clear to Send (CTS) hardware controls
		 */
		Software(2),

		/**
		 * Both the Request-to-Send (RTS) hardware control and the XON/XOFF software controls are used
		 */
		Both(3);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		Handshake(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Serial Communication Parity
	 */
	public enum Parity {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * No parity check occurs
		 */
		None(0),

		/**
		 * Sets the parity bit so that the count of bits set is an even number
		 */
		Even(1),

		/**
		 * Sets the parity bit so that the count of bits set is an odd number
		 */
		Odd(2);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		Parity(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * The default constructor will initialize the class with default values.
	 * <p>
	 * Initially all of the _IsValid parameters will be false because no data has been processed.  To 'populate' the values, the object must invoke <see cref="PrinterState.Update(ONeil.Connection.ConnectionBase, int)">Update</see> with the query response string.
	 */
	public SSP_CommParameters() {

		// Set Query Values
		m_QueryDescription = "SSP Comm Parameters";
		m_Query = "{S2?}";
		m_QueryResponseHeader = "{S2!";

		// Add names
		addName("B", "Baud Rate");
		addName("H", "RS232 Handshake");
		addName("N", "RS232 Data Bits");
		addName("P", "RS232 Parity");

		return;
	}
	/**
	 * Indicates if getBaudRate() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getBaudRate() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see SSP_CommParameters#getBaudRate()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getBaudRate_IsPresent() {
		return containsData("B") && isString("B");
	}

	/**
	 * Serial Baud Rate
	 * <p>
	 * The return value for this parameter is only valid if getBaudRate_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see SSP_CommParameters#getBaudRate_IsPresent()
	 * @return The current value for the parameter.
	 */
	public BaudValue getBaudRate() {
		BaudValue result = BaudValue.Unset;

		// Determine value
		if (!containsData("B")) {
			// Value not set
		}
		else if (queryResult("B").equals("012")) {
			// 1200 Baud
			result = BaudValue.Baud_1200;
		}
		else if (queryResult("B").equals("024")) {
			// 2400 Baud
			result = BaudValue.Baud_2400;
		}
		else if (queryResult("B").equals("048")) {
			// 4800 Baud
			result = BaudValue.Baud_4800;
		}
		else if (queryResult("B").equals("096")) {
			// 9600 Baud
			result = BaudValue.Baud_9600;
		}
		else if (queryResult("B").equals("192")) {
			// 19200 Baud
			result = BaudValue.Baud_19200;
		}
		else if (queryResult("B").equals("384")) {
			// 38400 Baud
			result = BaudValue.Baud_38400;
		}
		else if (queryResult("B").equals("576")) {
			// 57600 Baud
			result = BaudValue.Baud_57600;
		}
		else if (queryResult("B").equals("115")) {
			// 115200 Baud
			result = BaudValue.Baud_115200;
		}
		else if (queryResult("B").equals("230")) {
			// 230400 Baud
			result = BaudValue.Baud_230400;
		}
		else if (queryResult("B").equals("460")) {
			// 460800 Baud
			result = BaudValue.Baud_460800;
		}
		else if (queryResult("B").equals("921")) {
			// 921600 Baud
			result = BaudValue.Baud_921600;
		}
		else {
			// Value not yet in DB
			result = BaudValue.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getRS232Handshake() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRS232Handshake() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see SSP_CommParameters#getRS232Handshake()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRS232Handshake_IsPresent() {
		return containsData("H") && isString("H");
	}

	/**
	 * Handshake type
	 * <p>
	 * The return value for this parameter is only valid if getRS232Handshake_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see SSP_CommParameters#getRS232Handshake_IsPresent()
	 * @return The current value for the parameter.
	 */
	public Handshake getRS232Handshake() {
		Handshake result = Handshake.Unset;

		// Determine value
		if (!containsData("H")) {
			// Value not set
		}
		else if (queryResult("H").equals("N")) {
			// No control is used for the handshake
			result = Handshake.None;
		}
		else if (queryResult("H").equals("H")) {
			// Request-to-Send (RTS) hardware flow control is used. RTS signals that data is available for transmission. If the input buffer becomes full, the RTS line will be set to false. The RTS line will be set to true when more room becomes available in the input buffer
			result = Handshake.Hardware;
		}
		else if (queryResult("H").equals("S")) {
			// The XON/XOFF software control protocol is used. The XOFF control is sent to stop the transmission of data. The XON control is sent to resume the transmission. These software controls are used instead of Request to Send (RTS) and Clear to Send (CTS) hardware controls
			result = Handshake.Software;
		}
		else if (queryResult("H").equals("B")) {
			// Both the Request-to-Send (RTS) hardware control and the XON/XOFF software controls are used
			result = Handshake.Both;
		}
		else {
			// Value not yet in DB
			result = Handshake.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getRS232DataBits() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRS232DataBits() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see SSP_CommParameters#getRS232DataBits()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRS232DataBits_IsPresent() {
		return containsData("N") && isInteger("N");
	}

	/**
	 * Data Bits
	 * <p>
	 * The return value for this parameter is only valid if getRS232DataBits_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see SSP_CommParameters#getRS232DataBits_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getRS232DataBits() {
		return parse_long("N");
	}

	/**
	 * Indicates if getRS232Parity() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRS232Parity() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see SSP_CommParameters#getRS232Parity()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRS232Parity_IsPresent() {
		return containsData("P") && isString("P");
	}

	/**
	 * Serial Communication Parity
	 * <p>
	 * The return value for this parameter is only valid if getRS232Parity_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see SSP_CommParameters#getRS232Parity_IsPresent()
	 * @return The current value for the parameter.
	 */
	public Parity getRS232Parity() {
		Parity result = Parity.Unset;

		// Determine value
		if (!containsData("P")) {
			// Value not set
		}
		else if (queryResult("P").equals("N")) {
			// No parity check occurs
			result = Parity.None;
		}
		else if (queryResult("P").equals("E")) {
			// Sets the parity bit so that the count of bits set is an even number
			result = Parity.Even;
		}
		else if (queryResult("P").equals("O")) {
			// Sets the parity bit so that the count of bits set is an odd number
			result = Parity.Odd;
		}
		else {
			// Value not yet in DB
			result = Parity.Unknown;
		}

		return result; 
	}

}
