package datamaxoneil.printer.configuration;

import java.util.HashMap;
import datamaxoneil.connection.ConnectionBase;
import java.util.regex.Pattern;
import java.util.regex.Matcher;


/**
 * This is the base class for the printer state code. This will handle
 * configuration queries and settings and all of the shared routines for the
 * sub-classes that each handle a specific printer query.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
abstract public class PrinterState extends PrinterData {

	/**
	 * This will re-query the printer to update the values for this set of
	 * parameters. All current values will be replaced with the values on the
	 * printer.
	 * <p>
	 * Initially the PrinterState objects are empty. Use this to <I>update</I>
	 * them with the latest values from the printer. Unlink the other version of
	 * Update which takes the printer response string, this will handle all of
	 * the printer querying for you.
	 * 
	 * @param connection Connection object to talk to the printer with.
	 * @param timeout Maximum number of msec to wait for a response.
	 */
	public void update(ConnectionBase connection, int timeout) {

		// Query Values
		connection.clearReadBuffer();
		connection.write(getQuery());

		// Parse Values
		update(connection.read("}", timeout));

		return;
	}

	/**
	 * This will parse the query results and return the query results as a
	 * Hashmap name/NameValue objects.
	 * 
	 * @param queryResults Query results to process
	 * @return This will return the query results as a NameValue array. If this
	 *         query was invalid then an empty HashMap is returned.
	 * @throws IllegalStateException If a duplicate tag is found.
	 */
	protected HashMap<String, NameValue> querySettings(String queryResults) throws IllegalStateException {
		int index;
		Pattern regexObject;
		Matcher matchObject;
		HashMap<String, NameValue> settings = new HashMap<String, NameValue>();

		// Check if we need the MR? blank version
		if ((m_Query == "{MR?}") && 
			(queryResults.indexOf(m_QueryResponseHeader) == -1) && 
			(queryResults.indexOf("{N}") != -1)) {
			// Substitute in the blank version
			queryResults = "{MR!T1:|T2:|T3:}";
		}
		
		// Strip out to begging and end of this section
		index = queryResults.indexOf(m_QueryResponseHeader);
		if (index == -1)
			return settings;
		queryResults = queryResults.substring(index + m_QueryResponseHeader.length());

		index = queryResults.indexOf("}");
		if (index == -1)
			return settings;
		queryResults = queryResults.substring(0, index);

		// Check for the blank parameter (thanks for being consistent)
		if (m_Names.containsKey("")) {
			// Add the no parameter value
			settings.put("", new NameValue("", queryResults));
		}
		else {
			// Parse the settings
			if (m_Query.equalsIgnoreCase("{MR?}"))
				regexObject = Pattern.compile("([^:]+):([^|]*)\\|*");
			else
				regexObject = Pattern.compile("([^:]+):([^;]*);*");
			matchObject = regexObject.matcher(queryResults);

			while (matchObject.find()) {
				// We have a match so process section
				String name = matchObject.group(1);
				String value = matchObject.group(2);

				// Add to the list
				if (settings.containsKey(name)) {
					// Duplicate key?
					throw new IllegalStateException("Duplicate parameter '" + name
							+ "' found in query '" + m_Query + "': " + queryResults);
				}
				else if (m_Names.containsKey(name)) {
					// Its on the list
					settings.put(name, new NameValue(name, value));
				}
				else {
					// Unsupported Parameter
					settings.put(name, new NameValue(name, value));
				}
			}
		}

		return settings;
	}
}