package datamaxoneil.printer.configuration;

import java.util.List;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.regex.Pattern;
import java.util.regex.Matcher;


/**
 * This is the base class for the printer state code. This will handle
 * configuration queries and settings and all of the shared routines for the
 * sub-classes that each handle a specific printer query.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
abstract public class PrinterData {

	/**
	 * This class is used to store data items retrieved in printer queries. It
	 * is an internal class used as a structure.
	 */
	protected class NameValue {
		/** Name of the parameter */
		public String m_Name;

		/** Value of the parameter */
		public String m_Value;

		/** Name of the parameter 
		 * @return Name of the parameter */
		public String getName() {
			return m_Name;
		}

		/** Value of the parameter 
		 * @return Value of the parameter */
		public String getValue() {
			return m_Value;
		}

		/**
		 * Constructor that will set the read only values
		 * 
		 * @param name Name of the parameter
		 * @param value Value for the parameter
		 */
		public NameValue(String name, String value) {
			m_Name = name;
			m_Value = value;
			return;
		}
	};

	/**
	 * Names of the different values expected
	 */
	protected HashMap<String, String> m_Names = new HashMap<String, String>();

	/**
	 * Names of the different values expected in logical order
	 */
	private List<String> m_NamesOrder = new ArrayList<String>();

	/**
	 * Queried Name/Value pairs
	 */
	private HashMap<String, NameValue> m_NameValuePairs_Queried = new HashMap<String, NameValue>();

	/**
	 * Current Name/Value pairs
	 */
	private HashMap<String, NameValue> m_NameValuePairs_Current = new HashMap<String, NameValue>();

	/**
	 * Description of the query
	 */
	protected String m_QueryDescription;

	/**
	 * Query to retrieve data
	 */
	protected String m_Query;

	/**
	 * Query response header
	 */
	protected String m_QueryResponseHeader;

	/**
	 * This is true if the configuration information is valid. If the printer
	 * doesn't support the functionality or query failed this will return false.
	 * <p>
	 * Because not all queries are valid for all printer configurations this
	 * parameter is designed to indicate if the PrinterState object is populated
	 * with any values. A false value can mean either the object has never had
	 * Update run, or that when an update was performed nothing was returned
	 * from the printer.
	 * 
	 * @return If the configuration information is valid.
	 */
	public boolean getValid() {
		return (m_NameValuePairs_Queried.size() > 0);
	}

	/**
	 * This will return the query to send to the printer to retrieve the data.
	 * 
	 * @return The query to send to the printer to retrieve the data.
	 */
	public String getQuery() {
		return "\u001B" + m_Query;
	}

	/**
	 * This will return the description for the query.
	 * <p>
	 * The short description is useful for adding context to the retrieved
	 * results when you are displaying retrieved values.
	 * 
	 * @return The description for the query.
	 */
	public String getQueryDescription() {
		return m_QueryDescription;
	}

	/**
	 * This will parse the query response from the printer to update the values
	 * for this set of parameters. All current values will be replaced with the
	 * values on the printer.
	 * <p>
	 * Initially the PrinterState objects are empty. Use this to <I>update</I>
	 * them with the latest values from the printer. Unlink the other version of
	 * Update which takes the connection object to the printer and handles the
	 * querying for you, this version takes the query response. This allows the
	 * querying to be done at another time than processing.
	 * 
	 * @param responseString This is the string returned from the printer for
	 *        this query.
	 */
	public void update(String responseString) {

		// Query Values and store in queried and current
		m_NameValuePairs_Queried = querySettings(responseString);

		// Clone
		m_NameValuePairs_Current.clear();

		for (String key : m_NameValuePairs_Queried.keySet()) {
			m_NameValuePairs_Current.put(key, m_NameValuePairs_Queried.get(key));
		}

		return;
	}

	/**
	 * This will return a string value of the parameter token specified. If the
	 * token is not present then null is returned.
	 * <p>
	 * This function is good for generically accessing data of PrinterState
	 * objects.
	 * 
	 * @param name Name of the parameter to retrieve the value for.
	 * @return String with the value of the parameter.
	 */
	public String queryResult(String name) {
		String result = null;

		// Fetch
		if (m_NameValuePairs_Current.containsKey(name)) {
			// Present
			result = ((NameValue) m_NameValuePairs_Current.get(name)).getValue();
		}

		return result;
	}

	/**
	 * This will return an array list of the items returned in the query it the
	 * form of Name{tab}Value.
	 * <p>
	 * This function is good for generically accessing data of PrinterState
	 * objects. Altering these values does on alter their actual value.
	 * 
	 * @return ArrayList with the Name and Value items separated by a tab.
	 */
	public List<String> queryResults() {
		int index = 1;
		ArrayList<String> items = new ArrayList<String>();

		// Build a copy of knowns
		for (String name : m_NamesOrder) {

			// Check if is expected
			if (!m_NameValuePairs_Current.containsKey(name))
				continue;

			items.add((String) m_Names.get(name) + "\t"
					+ ((NameValue) m_NameValuePairs_Current.get(name)).getValue());
		}

		// Build a copy of unknowns
		for (NameValue item : m_NameValuePairs_Current.values()) {
			// Print out the values line by line
			if (!m_Names.containsKey(item.getName())) {
				// Unknown
				items.add("Unknown" + index + "\t" + item.getValue());
			}
		}

		return items;
	}

	/**
	 * Returns a String that represents the current configuration.
	 * 
	 * @return The current configuration data.
	 */
	public String toString() {
		return toString("L");
	}

	/**
	 * Returns a String that represents the current configuration.
	 * 
	 * @param format Format of the data to be returned. 'L' is for long, 'S' is
	 *        for short.
	 * @return The current configuration data.
	 */
	public String toString(String format) {
		String results = "";
		String resultsUnknown = "";

		// Print out line by line
		if ("L".compareTo(format) == 0) {
			// Long Format
			results += m_QueryDescription + "\n";

			for (String name : m_NamesOrder) {

				// Check if is expected
				if (!m_NameValuePairs_Current.containsKey(name))
					continue;

				results += (String) m_Names.get(name) + "("
						+ ((NameValue) m_NameValuePairs_Current.get(name)).getName() + ")" + ":"
						+ ((NameValue) m_NameValuePairs_Current.get(name)).getValue() + "\n";

			}

			// Unknowns
			for (NameValue item : m_NameValuePairs_Current.values()) {

				// Print out the values line by line
				if (!m_Names.containsKey(item.getName())) {
					// Unknown
					resultsUnknown += "\t*** Unknown Parameter(" + item.getName() + "): "
							+ item.getValue() + "\n";
				}
			}
		}
		else if ("S".compareTo(format) == 0) {
			// Short

			for (String name : m_NamesOrder) {

				// Check if is expected
				if (!m_NameValuePairs_Current.containsKey(name))
					continue;

				if (name != "") {
					results += ((NameValue) m_NameValuePairs_Current.get(name)).getName() + ": "
							+ ((NameValue) m_NameValuePairs_Current.get(name)).getValue() + "\n";
				}
				else {
					results += ((NameValue) m_NameValuePairs_Current.get(name)).getValue() + "\n";
				}
			}

			// Unknowns
			for (NameValue item : m_NameValuePairs_Current.values()) {

				// Print out the values line by line
				if (!m_Names.containsKey(item.getName())) {
					// Unknown
					resultsUnknown += item.getName() + ": " + item.getValue() + "\n";
				}
			}
		}
		else {
			// Unknown
			throw new IllegalArgumentException("toString parameter must be 'L' or 'S'");
		}

		return results + resultsUnknown;
	}

	/**
	 * This will parse the query results and return the query results as a
	 * Hashmap name/NameValue objects.
	 * 
	 * @param queryResults Query results to process
	 * @return This will return the query results as a NameValue array. If this
	 *         query was invalid then an empty HashMap is returned.
	 * @throws IllegalStateException If a duplicate tag is found.
	 */
	protected HashMap<String, NameValue> querySettings(String queryResults) throws IllegalStateException {
		Pattern regexObject;
		Matcher matchObject;
		HashMap<String, NameValue> settings = new HashMap<String, NameValue>();
	
		// Parse the settings
		regexObject = Pattern.compile("([^:]+):([^;]*);*");
		matchObject = regexObject.matcher(queryResults);

		while (matchObject.find()) {
			// We have a match so process section
			String name = matchObject.group(1);
			String value = matchObject.group(2);

			// Add to the list
			if (settings.containsKey(name)) {
				// Duplicate key?
				throw new IllegalStateException("Duplicate parameter '" + name
						+ "' found in query '" + m_Query + "': " + queryResults);
			}
			else if (m_Names.containsKey(name)) {
				// Its on the list
				settings.put(name, new NameValue(name, value));
			}
			else {
				// Unsupported Parameter
				settings.put(name, new NameValue(name, value));
			}
		}

		return settings;
	}

	/**
	 * Indicates if we have the parameter value.
	 * 
	 * @param parameter Parameter key
	 * @return True if the parameter is present, false otherwise.
	 */
	protected boolean containsData(String parameter) {
		return m_NameValuePairs_Current.containsKey(parameter);
	}

	/**
	 * Return if this is a valid integer value
	 *
	 * @param name Parameter key
	 * @return True if this is a valid integer
	 */
	protected boolean isInteger(String name) {
		int index = 0;
		String temp = m_NameValuePairs_Current.get(name).getValue().trim();
		
		// Trim _trailing_ non digits
		while (
			(index < temp.length()) && 
			(((temp.charAt(index) >='0') && (temp.charAt(index) <= '9')) || (temp.charAt(index) == '-'))
			) {
			index++;
		}
		
		// Check for nothing
		if (index == 0) return false;
		
		// Convert
		try {Integer.parseInt(temp.substring(0, index));}
		catch (Exception ignore){return false;}
		return (temp != null) && (temp.length() > 0);
	}

	/**
	 * Return if this is a valid double value
	 *
	 * @param name Parameter key
	 * @return True if this is a valid double
	 */
	protected boolean isDouble(String name) {
		int index = 0;
		String temp = m_NameValuePairs_Current.get(name).getValue().trim();
		
		// Trim _trailing_ non digits
		while (
			(index < temp.length()) && 
			(((temp.charAt(index) >='0') && (temp.charAt(index) <= '9')) || (temp.charAt(index) == '-') || (temp.charAt(index) == '.'))
			) {
			index++;
		}
		
		// Check for nothing
		if (index == 0) return false;
		
		try {Double.parseDouble(temp.substring(0, index));}
		catch (Exception ignore){return false;}
		return (temp != null) && (temp.length() > 0);
	}

	/**
	 * Return if this is a valid string value
	 *
	 * @param name Parameter key
	 * @return True if this is a valid string
	 */
	protected boolean isString(String name) {
		String temp = m_NameValuePairs_Current.get(name).getValue();
		return (temp != null) && (temp.length() > 0);
	}

	/**
	 * This will return the value of the parameter in the retrieved values.
	 * 
	 * @param name Parameter name
	 * @param trueValue Value used when this is true
	 * @param falseValue Value used when this is false
	 * @return Value of the parameter.
	 * @throws IllegalArgumentException If the value is not one of the two
	 *         expected values.
	 */
	protected boolean parse_boolean(String name, String trueValue, String falseValue) throws IllegalArgumentException {
		boolean result = false;

		// Check for existence
		if (m_NameValuePairs_Current.containsKey(name)) {
			// Fetch Value
			String value = ((NameValue) m_NameValuePairs_Current.get(name)).getValue();

			// Convert to a boolean
			if (value.equals(trueValue)) {
				// True
				result = true;
			}
			else if (value.equals(falseValue)) {
				// False
				result = false;
			}
			else {
				// Not what we expected
				throw new IllegalArgumentException("Boolean value should have been \"" + trueValue
						+ "\" or \"" + falseValue + "\" but instead was \"" + value + "\".");
			}
		}

		return result;
	}

	/**
	 * This will return the value of the parameter in the retrieved values.
	 * 
	 * @param name Parameter name
	 * @return Value of the parameter.
	 */
	protected long parse_long(String name) {
		long result = 0;

		// Check for existence
		if (m_NameValuePairs_Current.containsKey(name)) {
			// Fetch Value
			String temp = ((NameValue) m_NameValuePairs_Current.get(name)).getValue();

			// Trim _trailing_ non digits
			int index = 0;
			while (
				(index < temp.length()) && 
				(((temp.charAt(index) >='0') && (temp.charAt(index) <= '9')) || (temp.charAt(index) == '-'))
				) {
				index++;
			}
			temp = temp.substring(0, index);

			// Convert to a long
			try {
				result = Integer.parseInt(temp);
			}
			catch (Exception ignore) {
			}
		}

		return result;

	}

	/**
	 * This will return the value of the parameter in the retrieved values.
	 * 
	 * @param name Parameter name
	 * @return Value of the parameter.
	 */
	protected double parse_double(String name) {
		double result = 0.0;

		// Check for existence
		if (m_NameValuePairs_Current.containsKey(name)) {
			// Fetch Value
			String temp = ((NameValue) m_NameValuePairs_Current.get(name)).getValue();

			// Strip out training non-numbers
			int index = 0;
			while (
					(index < temp.length()) && 
					(((temp.charAt(index) >='0') && (temp.charAt(index) <= '9')) || (temp.charAt(index) == '-') || (temp.charAt(index) == '.'))
					) {
					index++;
			}
			
			temp = temp.substring(0, index);

			// Convert to a double
			try {
				result = Double.parseDouble(temp);
			}
			catch (Exception ignore) {
			}
		}

		return result;
	}

	/**
	 * This will return the value of the parameter in the retrieved values.
	 * 
	 * @param name Parameter name
	 * @return Value of the parameter.
	 */
	protected String parse_string(String name) {
		String result = null;

		// Check for existence
		if (m_NameValuePairs_Current.containsKey(name)) {
			// Fetch Value
			result = ((NameValue) m_NameValuePairs_Current.get(name)).getValue();
		}

		return result;
	}

	/**
	 * Add this parameter to the internal list.
	 * 
	 * @param parameter Parameter name
	 * @param description Parameter description
	 */
	protected void addName(String parameter, String description) {

		// Add to Hash
		m_Names.put(parameter, description);

		// Add to order list
		m_NamesOrder.add(parameter);

		return;
	}
}