package datamaxoneil.printer.configuration;

/**
 * This class handles the data from an {CL?} query as well as
 * accessors to read the data values.
 * <p>
 * This class supports the following possible parameters:
 * <table border=1>
 *    <tr><th>Parameter</th><th>Description</th></tr>
 *    <tr><td>B</td><td>BackUp Distance</td></tr>
 *    <tr><td>BKO</td><td>Backup Offset</td></tr>
 *    <tr><td>D</td><td>How far past the QMark should we stop</td></tr>
 *    <tr><td>I</td><td>Additional self-tests to print</td></tr>
 *    <tr><td>M</td><td>Maximum length to look for QMark</td></tr>
 *    <tr><td>MO</td><td>Horizontal Offset</td></tr>
 *    <tr><td>P</td><td>Use Presenter</td></tr>
 *    <tr><td>QBO</td><td>QMARKB Offset</td></tr>
 *    <tr><td>QGO</td><td>QMARKG Offset</td></tr>
 *    <tr><td>QTO</td><td>QMARKT Offset</td></tr>
 *    <tr><td>QW</td><td>White QMark</td></tr>
 *    <tr><td>S</td><td>Paperout Sensor</td></tr>
 *    <tr><td>T</td><td>Type of paper stock used</td></tr>
 *    <tr><td>U</td><td>Label under presenter timeout</td></tr>
 *    <tr><td>W</td><td>Auto QMark Advance</td></tr>
 *    <tr><td>WB</td><td>Auto QMark Backup</td></tr>
 * </table>
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class LabelConfiguration extends PrinterState {

	/**
	 * Paperout Sensor
	 */
	public enum PaperOutSensorValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * Use the bottom, non-printing side of paper, to detect paper out.
		 */
		Bottom(66),

		/**
		 * Use the top, printing side of paper, to detect paper out.
		 */
		Top(84);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		PaperOutSensorValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Type of paper stock used
	 */
	public enum QMarkValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * Queue mark is on the bottom, side that is not printed on, of the paper.  
		 */
		BottomQMark(66),

		/**
		 * Instead of a printed queue mark, this is a transparent area between labels that allows light to pass through.
		 */
		InterLabelGap(71),

		/**
		 * Plain paper, no queue mark
		 */
		PlainPaper(80),

		/**
		 * Queue mark is on the top, side that is printed on, of the paper.  
		 */
		TopQMARK(84);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		QMarkValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * The default constructor will initialize the class with default values.
	 * <p>
	 * Initially all of the _IsValid parameters will be false because no data has been processed.  To 'populate' the values, the object must invoke <see cref="PrinterState.Update(ONeil.Connection.ConnectionBase, int)">Update</see> with the query response string.
	 */
	public LabelConfiguration() {

		// Set Query Values
		m_QueryDescription = "Label Related Configuration Values";
		m_Query = "{CL?}";
		m_QueryResponseHeader = "{CL!";

		// Add names
		addName("B", "BackUp Distance");
		addName("BKO", "Backup Offset");
		addName("D", "QMark Stop Length");
		addName("I", "Additional Self Test Prints");
		addName("M", "Maximum QMark Advance");
		addName("MO", "Horizontal Offset");
		addName("P", "Use Presenter");
		addName("QBO", "QMARKB Offset");
		addName("QGO", "QMARKG Offset");
		addName("QTO", "QMARKT Offset");
		addName("QW", "White QMark");
		addName("S", "Paperout Sensor");
		addName("T", "Paper Stock Type");
		addName("U", "Presenter Timeout");
		addName("W", "Auto QMark Advance");
		addName("WB", "Auto QMark Backup");

		return;
	}
	/**
	 * Indicates if getBackUpDistance() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getBackUpDistance() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getBackUpDistance()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getBackUpDistance_IsPresent() {
		return containsData("B") && isInteger("B");
	}

	/**
	 * BackUp Distance
	 * <p>
	 * The return value for this parameter is only valid if getBackUpDistance_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getBackUpDistance_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getBackUpDistance() {
		return parse_long("B");
	}

	/**
	 * Indicates if getBackupOffset() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getBackupOffset() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getBackupOffset()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getBackupOffset_IsPresent() {
		return containsData("BKO") && isInteger("BKO");
	}

	/**
	 * Backup Offset
	 * <p>
	 * The return value for this parameter is only valid if getBackupOffset_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getBackupOffset_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getBackupOffset() {
		return parse_long("BKO");
	}

	/**
	 * Indicates if getQMarkStopLength() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getQMarkStopLength() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getQMarkStopLength()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getQMarkStopLength_IsPresent() {
		return containsData("D") && isInteger("D");
	}

	/**
	 * How far past the QMark should we stop
	 * <p>
	 * The return value for this parameter is only valid if getQMarkStopLength_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getQMarkStopLength_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getQMarkStopLength() {
		return parse_long("D");
	}

	/**
	 * Indicates if getAdditionalSelfTestPrints() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getAdditionalSelfTestPrints() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getAdditionalSelfTestPrints()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getAdditionalSelfTestPrints_IsPresent() {
		return containsData("I") && isInteger("I");
	}

	/**
	 * Additional self-tests to print
	 * <p>
	 * The return value for this parameter is only valid if getAdditionalSelfTestPrints_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getAdditionalSelfTestPrints_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getAdditionalSelfTestPrints() {
		return parse_long("I");
	}

	/**
	 * Indicates if getMaximumQMarkAdvance() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getMaximumQMarkAdvance() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getMaximumQMarkAdvance()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getMaximumQMarkAdvance_IsPresent() {
		return containsData("M") && isInteger("M");
	}

	/**
	 * Maximum length to look for QMark
	 * <p>
	 * The return value for this parameter is only valid if getMaximumQMarkAdvance_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getMaximumQMarkAdvance_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getMaximumQMarkAdvance() {
		return parse_long("M");
	}

	/**
	 * Indicates if getHorizontalOffset() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getHorizontalOffset() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getHorizontalOffset()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getHorizontalOffset_IsPresent() {
		return containsData("MO") && isInteger("MO");
	}

	/**
	 * Horizontal Offset
	 * <p>
	 * The return value for this parameter is only valid if getHorizontalOffset_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getHorizontalOffset_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getHorizontalOffset() {
		return parse_long("MO");
	}

	/**
	 * Indicates if getUsePresenter() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getUsePresenter() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getUsePresenter()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getUsePresenter_IsPresent() {
		return containsData("P") && isString("P");
	}

	/**
	 * Use Presenter
	 * <p>
	 * The return value for this parameter is only valid if getUsePresenter_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getUsePresenter_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getUsePresenter() {
		return parse_boolean("P", "Y", "N");
	}

	/**
	 * Indicates if getQMARKBOffset() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getQMARKBOffset() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getQMARKBOffset()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getQMARKBOffset_IsPresent() {
		return containsData("QBO") && isInteger("QBO");
	}

	/**
	 * QMARKB Offset
	 * <p>
	 * The return value for this parameter is only valid if getQMARKBOffset_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getQMARKBOffset_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getQMARKBOffset() {
		return parse_long("QBO");
	}

	/**
	 * Indicates if getQMARKGOffset() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getQMARKGOffset() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getQMARKGOffset()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getQMARKGOffset_IsPresent() {
		return containsData("QGO") && isInteger("QGO");
	}

	/**
	 * QMARKG Offset
	 * <p>
	 * The return value for this parameter is only valid if getQMARKGOffset_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getQMARKGOffset_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getQMARKGOffset() {
		return parse_long("QGO");
	}

	/**
	 * Indicates if getQMARKTOffset() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getQMARKTOffset() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getQMARKTOffset()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getQMARKTOffset_IsPresent() {
		return containsData("QTO") && isInteger("QTO");
	}

	/**
	 * QMARKT Offset
	 * <p>
	 * The return value for this parameter is only valid if getQMARKTOffset_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getQMARKTOffset_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getQMARKTOffset() {
		return parse_long("QTO");
	}

	/**
	 * Indicates if getWhiteQMark() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getWhiteQMark() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getWhiteQMark()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getWhiteQMark_IsPresent() {
		return containsData("QW") && isString("QW");
	}

	/**
	 * White QMark
	 * <p>
	 * The return value for this parameter is only valid if getWhiteQMark_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getWhiteQMark_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getWhiteQMark() {
		return parse_boolean("QW", "Y", "N");
	}

	/**
	 * Indicates if getPaperoutSensor() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPaperoutSensor() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getPaperoutSensor()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPaperoutSensor_IsPresent() {
		return containsData("S") && isString("S");
	}

	/**
	 * Paperout Sensor
	 * <p>
	 * The return value for this parameter is only valid if getPaperoutSensor_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getPaperoutSensor_IsPresent()
	 * @return The current value for the parameter.
	 */
	public PaperOutSensorValue getPaperoutSensor() {
		PaperOutSensorValue result = PaperOutSensorValue.Unset;

		// Determine value
		if (!containsData("S")) {
			// Value not set
		}
		else if (queryResult("S").equals("B")) {
			// Use the bottom, non-printing side of paper, to detect paper out.
			result = PaperOutSensorValue.Bottom;
		}
		else if (queryResult("S").equals("T")) {
			// Use the top, printing side of paper, to detect paper out.
			result = PaperOutSensorValue.Top;
		}
		else {
			// Value not yet in DB
			result = PaperOutSensorValue.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getPaperStockType() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPaperStockType() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getPaperStockType()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPaperStockType_IsPresent() {
		return containsData("T") && isString("T");
	}

	/**
	 * Type of paper stock used
	 * <p>
	 * The return value for this parameter is only valid if getPaperStockType_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getPaperStockType_IsPresent()
	 * @return The current value for the parameter.
	 */
	public QMarkValue getPaperStockType() {
		QMarkValue result = QMarkValue.Unset;

		// Determine value
		if (!containsData("T")) {
			// Value not set
		}
		else if (queryResult("T").equals("B")) {
			// Queue mark is on the bottom, side that is not printed on, of the paper.  
			result = QMarkValue.BottomQMark;
		}
		else if (queryResult("T").equals("G")) {
			// Instead of a printed queue mark, this is a transparent area between labels that allows light to pass through.
			result = QMarkValue.InterLabelGap;
		}
		else if (queryResult("T").equals("P")) {
			// Plain paper, no queue mark
			result = QMarkValue.PlainPaper;
		}
		else if (queryResult("T").equals("T")) {
			// Queue mark is on the top, side that is printed on, of the paper.  
			result = QMarkValue.TopQMARK;
		}
		else {
			// Value not yet in DB
			result = QMarkValue.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getPresenterTimeout() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPresenterTimeout() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getPresenterTimeout()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPresenterTimeout_IsPresent() {
		return containsData("U") && isInteger("U");
	}

	/**
	 * Label under presenter timeout
	 * <p>
	 * The return value for this parameter is only valid if getPresenterTimeout_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getPresenterTimeout_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getPresenterTimeout() {
		return parse_long("U");
	}

	/**
	 * Indicates if getAutoQMarkAdvance() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getAutoQMarkAdvance() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getAutoQMarkAdvance()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getAutoQMarkAdvance_IsPresent() {
		return containsData("W") && isString("W");
	}

	/**
	 * Auto QMark Advance
	 * <p>
	 * The return value for this parameter is only valid if getAutoQMarkAdvance_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getAutoQMarkAdvance_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getAutoQMarkAdvance() {
		return parse_boolean("W", "Y", "N");
	}

	/**
	 * Indicates if getAutoQMarkBackup() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getAutoQMarkBackup() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getAutoQMarkBackup()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getAutoQMarkBackup_IsPresent() {
		return containsData("WB") && isString("WB");
	}

	/**
	 * Auto QMark Backup
	 * <p>
	 * The return value for this parameter is only valid if getAutoQMarkBackup_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see LabelConfiguration#getAutoQMarkBackup_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getAutoQMarkBackup() {
		return parse_boolean("WB", "Y", "N");
	}

}
