package datamaxoneil.printer.configuration;

/**
 * This class handles the data from an {IR?} query as well as
 * accessors to read the data values.
 * <p>
 * This class supports the following possible parameters:
 * <table border=1>
 *    <tr><th>Parameter</th><th>Description</th></tr>
 *    <tr><td>DV</td><td>Direct Version</td></tr>
 *    <tr><td>ID</td><td>IrDA Nickname</td></tr>
 *    <tr><td>IN</td><td>IrDA Name</td></tr>
 *    <tr><td>IV</td><td>IrDA Version</td></tr>
 *    <tr><td>P</td><td>IR Protocol</td></tr>
 * </table>
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class IrDAConfiguration extends PrinterState {

	/**
	 * IR Protocol
	 */
	public enum IrDAProtocolValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * IrDA Off
		 */
		OFF(0),

		/**
		 * No IrDA present
		 */
		None(1),

		/**
		 * Standard Protocol
		 */
		IrDA(2),

		/**
		 * Amplitude Shift Key
		 */
		ASK(3),

		/**
		 * Amplitude Shift Key + CRC
		 */
		ASK_CRC(4),

		/**
		 * Direct
		 */
		Direct(5),

		/**
		 * Direct + CRC
		 */
		Direct_CRC(6),

		/**
		 * Pulse
		 */
		Pulse(7),

		/**
		 * Pulse + CRC
		 */
		Pulse_CRC(8);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		IrDAProtocolValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * The default constructor will initialize the class with default values.
	 * <p>
	 * Initially all of the _IsValid parameters will be false because no data has been processed.  To 'populate' the values, the object must invoke <see cref="PrinterState.Update(ONeil.Connection.ConnectionBase, int)">Update</see> with the query response string.
	 */
	public IrDAConfiguration() {

		// Set Query Values
		m_QueryDescription = "IrDA Configuration";
		m_Query = "{IR?}";
		m_QueryResponseHeader = "{IR!";

		// Add names
		addName("DV", "Direct Version");
		addName("ID", "IrDA Nickname");
		addName("IN", "IrDA Name");
		addName("IV", "IrDA Version");
		addName("P", "Protocol");

		return;
	}
	/**
	 * Indicates if getDirectVersion() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getDirectVersion() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getDirectVersion()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getDirectVersion_IsPresent() {
		return containsData("DV") && isString("DV");
	}

	/**
	 * Direct Version
	 * <p>
	 * The return value for this parameter is only valid if getDirectVersion_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getDirectVersion_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getDirectVersion() {
		return parse_string("DV");
	}

	/**
	 * Indicates if getIrDANickname() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getIrDANickname() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getIrDANickname()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getIrDANickname_IsPresent() {
		return containsData("ID") && isString("ID");
	}

	/**
	 * IrDA Nickname
	 * <p>
	 * The return value for this parameter is only valid if getIrDANickname_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getIrDANickname_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getIrDANickname() {
		return parse_string("ID");
	}

	/**
	 * Indicates if getIrDAName() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getIrDAName() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getIrDAName()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getIrDAName_IsPresent() {
		return containsData("IN") && isString("IN");
	}

	/**
	 * IrDA Name
	 * <p>
	 * The return value for this parameter is only valid if getIrDAName_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getIrDAName_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getIrDAName() {
		return parse_string("IN");
	}

	/**
	 * Indicates if getIrDAVersion() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getIrDAVersion() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getIrDAVersion()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getIrDAVersion_IsPresent() {
		return containsData("IV") && isString("IV");
	}

	/**
	 * IrDA Version
	 * <p>
	 * The return value for this parameter is only valid if getIrDAVersion_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getIrDAVersion_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getIrDAVersion() {
		return parse_string("IV");
	}

	/**
	 * Indicates if getProtocol() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getProtocol() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getProtocol()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getProtocol_IsPresent() {
		return containsData("P") && isString("P");
	}

	/**
	 * IR Protocol
	 * <p>
	 * The return value for this parameter is only valid if getProtocol_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see IrDAConfiguration#getProtocol_IsPresent()
	 * @return The current value for the parameter.
	 */
	public IrDAProtocolValue getProtocol() {
		IrDAProtocolValue result = IrDAProtocolValue.Unset;

		// Determine value
		if (!containsData("P")) {
			// Value not set
		}
		else if (queryResult("P").equals("OFF")) {
			// IrDA Off
			result = IrDAProtocolValue.OFF;
		}
		else if (queryResult("P").equals("None")) {
			// No IrDA present
			result = IrDAProtocolValue.None;
		}
		else if (queryResult("P").equals("IrDA")) {
			// Standard Protocol
			result = IrDAProtocolValue.IrDA;
		}
		else if (queryResult("P").equals("ASK")) {
			// Amplitude Shift Key
			result = IrDAProtocolValue.ASK;
		}
		else if (queryResult("P").equals("ASK_CRC")) {
			// Amplitude Shift Key + CRC
			result = IrDAProtocolValue.ASK_CRC;
		}
		else if (queryResult("P").equals("Direct")) {
			// Direct
			result = IrDAProtocolValue.Direct;
		}
		else if (queryResult("P").equals("Direct_CRC")) {
			// Direct + CRC
			result = IrDAProtocolValue.Direct_CRC;
		}
		else if (queryResult("P").equals("Pulse")) {
			// Pulse
			result = IrDAProtocolValue.Pulse;
		}
		else if (queryResult("P").equals("Pulse_CRC")) {
			// Pulse + CRC
			result = IrDAProtocolValue.Pulse_CRC;
		}
		else {
			// Value not yet in DB
			result = IrDAProtocolValue.Unknown;
		}

		return result; 
	}

}
