package datamaxoneil.printer.configuration;

/**
 * This class handles the data from an {ST?} query as well as
 * accessors to read the data values.
 * <p>
 * This class supports the following possible parameters:
 * <table border=1>
 *    <tr><th>Parameter</th><th>Description</th></tr>
 *    <tr><td>B</td><td>Battery Temperature and Voltage Status</td></tr>
 *    <tr><td>E</td><td>Printer Error Status</td></tr>
 *    <tr><td>J</td><td>Paper Jam</td></tr>
 *    <tr><td>L</td><td>Head Lever Down</td></tr>
 *    <tr><td>P</td><td>Paper Present</td></tr>
 *    <tr><td>R</td><td>Remaining Ram</td></tr>
 *    <tr><td>S</td><td>Printer Status</td></tr>
 * </table>
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class GeneralStatus extends PrinterState {

	/**
	 * Battery Temperature and Voltage Status
	 */
	public enum BatteryStatusValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * Battery is in good condition.
		 */
		AllOK(79),

		/**
		 * Battery temperature is out of range.
		 */
		TemperatureError(84),

		/**
		 * Battery voltage is out of range.
		 */
		VoltageError(86);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		BatteryStatusValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Printer Error Status
	 */
	public enum ErrorStatusValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * No error
		 */
		None(78),

		/**
		 * Invalid command
		 */
		Command(99),

		/**
		 * Invalid data
		 */
		Data(100),

		/**
		 * Invalid global parameter
		 */
		GlobalParameter(103),

		/**
		 * Invalid name
		 */
		Name(110),

		/**
		 * Invalid protocol
		 */
		Protocol(112),

		/**
		 * Invalid syntax
		 */
		Syntax(115),

		/**
		 * Invalid graphics file
		 */
		PCX_File(120);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		ErrorStatusValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Printer Status
	 */
	public enum PrinterStatusValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * Print job has completed.  This is only present when job statusing is turned on.
		 */
		Complete(67),

		/**
		 * Currently idle
		 */
		Idle(73),

		/**
		 * Print task killed
		 */
		Killed(75),

		/**
		 * Currently Printing
		 */
		Printing(80),

		/**
		 * Printer timed out
		 */
		TimeOut(84);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		PrinterStatusValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * The default constructor will initialize the class with default values.
	 * <p>
	 * Initially all of the _IsValid parameters will be false because no data has been processed.  To 'populate' the values, the object must invoke <see cref="PrinterState.Update(ONeil.Connection.ConnectionBase, int)">Update</see> with the query response string.
	 */
	public GeneralStatus() {

		// Set Query Values
		m_QueryDescription = "General Status";
		m_Query = "{ST?}";
		m_QueryResponseHeader = "{ST!";

		// Add names
		addName("B", "Battery Temp and Voltage Status");
		addName("E", "Error Status");
		addName("J", "Paper Jam");
		addName("L", "Head Lever Down");
		addName("P", "Paper Present");
		addName("R", "Remaining RAM");
		addName("S", "Printer Status");

		return;
	}
	/**
	 * Indicates if getBatteryTempandVoltageStatus() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getBatteryTempandVoltageStatus() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getBatteryTempandVoltageStatus()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getBatteryTempandVoltageStatus_IsPresent() {
		return containsData("B") && isString("B");
	}

	/**
	 * Battery Temperature and Voltage Status
	 * <p>
	 * The return value for this parameter is only valid if getBatteryTempandVoltageStatus_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getBatteryTempandVoltageStatus_IsPresent()
	 * @return The current value for the parameter.
	 */
	public BatteryStatusValue getBatteryTempandVoltageStatus() {
		BatteryStatusValue result = BatteryStatusValue.Unset;

		// Determine value
		if (!containsData("B")) {
			// Value not set
		}
		else if (queryResult("B").equals("O")) {
			// Battery is in good condition.
			result = BatteryStatusValue.AllOK;
		}
		else if (queryResult("B").equals("T")) {
			// Battery temperature is out of range.
			result = BatteryStatusValue.TemperatureError;
		}
		else if (queryResult("B").equals("V")) {
			// Battery voltage is out of range.
			result = BatteryStatusValue.VoltageError;
		}
		else {
			// Value not yet in DB
			result = BatteryStatusValue.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getErrorStatus() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getErrorStatus() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getErrorStatus()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getErrorStatus_IsPresent() {
		return containsData("E") && isString("E");
	}

	/**
	 * Printer Error Status
	 * <p>
	 * The return value for this parameter is only valid if getErrorStatus_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getErrorStatus_IsPresent()
	 * @return The current value for the parameter.
	 */
	public ErrorStatusValue getErrorStatus() {
		ErrorStatusValue result = ErrorStatusValue.Unset;

		// Determine value
		if (!containsData("E")) {
			// Value not set
		}
		else if (queryResult("E").equals("N")) {
			// No error
			result = ErrorStatusValue.None;
		}
		else if (queryResult("E").equals("c")) {
			// Invalid command
			result = ErrorStatusValue.Command;
		}
		else if (queryResult("E").equals("d")) {
			// Invalid data
			result = ErrorStatusValue.Data;
		}
		else if (queryResult("E").equals("g")) {
			// Invalid global parameter
			result = ErrorStatusValue.GlobalParameter;
		}
		else if (queryResult("E").equals("n")) {
			// Invalid name
			result = ErrorStatusValue.Name;
		}
		else if (queryResult("E").equals("p")) {
			// Invalid protocol
			result = ErrorStatusValue.Protocol;
		}
		else if (queryResult("E").equals("s")) {
			// Invalid syntax
			result = ErrorStatusValue.Syntax;
		}
		else if (queryResult("E").equals("x")) {
			// Invalid graphics file
			result = ErrorStatusValue.PCX_File;
		}
		else {
			// Value not yet in DB
			result = ErrorStatusValue.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getPaperJam() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPaperJam() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getPaperJam()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPaperJam_IsPresent() {
		return containsData("J") && isString("J");
	}

	/**
	 * Paper Jam
	 * <p>
	 * The return value for this parameter is only valid if getPaperJam_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getPaperJam_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getPaperJam() {
		return parse_boolean("J", "J", "N");
	}

	/**
	 * Indicates if getHeadLeverDown() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getHeadLeverDown() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getHeadLeverDown()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getHeadLeverDown_IsPresent() {
		return containsData("L") && isString("L");
	}

	/**
	 * Head Lever Down
	 * <p>
	 * The return value for this parameter is only valid if getHeadLeverDown_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getHeadLeverDown_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getHeadLeverDown() {
		return parse_boolean("L", "D", "U");
	}

	/**
	 * Indicates if getPaperPresent() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPaperPresent() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getPaperPresent()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPaperPresent_IsPresent() {
		return containsData("P") && isString("P");
	}

	/**
	 * Paper Present
	 * <p>
	 * The return value for this parameter is only valid if getPaperPresent_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getPaperPresent_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getPaperPresent() {
		return parse_boolean("P", "P", "N");
	}

	/**
	 * Indicates if getRemainingRAM() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRemainingRAM() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getRemainingRAM()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRemainingRAM_IsPresent() {
		return containsData("R") && isInteger("R");
	}

	/**
	 * Remaining Ram
	 * <p>
	 * The return value for this parameter is only valid if getRemainingRAM_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getRemainingRAM_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getRemainingRAM() {
		return parse_long("R");
	}

	/**
	 * Indicates if getPrinterStatus() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPrinterStatus() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getPrinterStatus()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPrinterStatus_IsPresent() {
		return containsData("S") && isString("S");
	}

	/**
	 * Printer Status
	 * <p>
	 * The return value for this parameter is only valid if getPrinterStatus_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralStatus#getPrinterStatus_IsPresent()
	 * @return The current value for the parameter.
	 */
	public PrinterStatusValue getPrinterStatus() {
		PrinterStatusValue result = PrinterStatusValue.Unset;

		// Determine value
		if (!containsData("S")) {
			// Value not set
		}
		else if (queryResult("S").equals("C")) {
			// Print job has completed.  This is only present when job statusing is turned on.
			result = PrinterStatusValue.Complete;
		}
		else if (queryResult("S").equals("I")) {
			// Currently idle
			result = PrinterStatusValue.Idle;
		}
		else if (queryResult("S").equals("K")) {
			// Print task killed
			result = PrinterStatusValue.Killed;
		}
		else if (queryResult("S").equals("P")) {
			// Currently Printing
			result = PrinterStatusValue.Printing;
		}
		else if (queryResult("S").equals("T")) {
			// Printer timed out
			result = PrinterStatusValue.TimeOut;
		}
		else {
			// Value not yet in DB
			result = PrinterStatusValue.Unknown;
		}

		return result; 
	}

}
