package datamaxoneil.printer.configuration;

/**
 * This class handles the data from an {CF?} query as well as
 * accessors to read the data values.
 * <p>
 * This class supports the following possible parameters:
 * <table border=1>
 *    <tr><th>Parameter</th><th>Description</th></tr>
 *    <tr><td>A</td><td>White Space Advance Configuration</td></tr>
 *    <tr><td>B</td><td>Serial Baud Rate</td></tr>
 *    <tr><td>D</td><td>Printing Darkness Adjustment</td></tr>
 *    <tr><td>F</td><td>Form Feed Enabled</td></tr>
 *    <tr><td>G</td><td>Beep When Charger Connected</td></tr>
 *    <tr><td>H</td><td>Handshake type</td></tr>
 *    <tr><td>I</td><td>Lines Per Page</td></tr>
 *    <tr><td>J</td><td>Job Status Report</td></tr>
 *    <tr><td>L</td><td>Default Printing Protocol</td></tr>
 *    <tr><td>LT</td><td>Self Test Print Language</td></tr>
 *    <tr><td>M</td><td>Form Feed Centering</td></tr>
 *    <tr><td>N</td><td>Data Bits</td></tr>
 *    <tr><td>P</td><td>Serial Communication Parity</td></tr>
 *    <tr><td>PF</td><td>Formfeed Button Disabled</td></tr>
 *    <tr><td>PO</td><td>Power Button Disabled</td></tr>
 *    <tr><td>PR</td><td>RF Button Disabled</td></tr>
 *    <tr><td>QC</td><td>Number of times to add .25 volts or 0x19 to the QMark volt threshold.  The default is 0 and as this number increases the QMark is more difficult to detect, as it decreases the QMark is easier to detect.</td></tr>
 *    <tr><td>R</td><td>RF Power Timeout</td></tr>
 *    <tr><td>S</td><td>Sound Enabled</td></tr>
 *    <tr><td>T</td><td>System Timeout</td></tr>
 *    <tr><td>TP</td><td>Special Test Print Mode</td></tr>
 *    <tr><td>U</td><td>Paper Out Beep</td></tr>
 *    <tr><td>UC</td><td>What USB device class is the printer currently configured for.</td></tr>
 *    <tr><td>USB</td><td>Is USB the current communication method.</td></tr>
 * </table>
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class GeneralConfiguration extends PrinterState {

	/**
	 * Serial Baud Rate
	 */
	public enum BaudValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * 1200 Baud
		 */
		Baud_1200(1200),

		/**
		 * 2400 Baud
		 */
		Baud_2400(2400),

		/**
		 * 4800 Baud
		 */
		Baud_4800(4800),

		/**
		 * 9600 Baud
		 */
		Baud_9600(9600),

		/**
		 * 19200 Baud
		 */
		Baud_19200(19200),

		/**
		 * 38400 Baud
		 */
		Baud_38400(38400),

		/**
		 * 57600 Baud
		 */
		Baud_57600(57600),

		/**
		 * 115200 Baud
		 */
		Baud_115200(115200),

		/**
		 * 230400 Baud
		 */
		Baud_230400(230400),

		/**
		 * 460800 Baud
		 */
		Baud_460800(460800),

		/**
		 * 921600 Baud
		 */
		Baud_921600(921600);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		BaudValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Handshake type
	 */
	public enum Handshake {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * No control is used for the handshake
		 */
		None(0),

		/**
		 * Request-to-Send (RTS) hardware flow control is used. RTS signals that data is available for transmission. If the input buffer becomes full, the RTS line will be set to false. The RTS line will be set to true when more room becomes available in the input buffer
		 */
		Hardware(1),

		/**
		 * The XON/XOFF software control protocol is used. The XOFF control is sent to stop the transmission of data. The XON control is sent to resume the transmission. These software controls are used instead of Request to Send (RTS) and Clear to Send (CTS) hardware controls
		 */
		Software(2),

		/**
		 * Both the Request-to-Send (RTS) hardware control and the XON/XOFF software controls are used
		 */
		Both(3);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		Handshake(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Default Printing Protocol
	 */
	public enum EmulationProtocolValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * PK80 Impact Printer Emulation
		 */
		PK80(13),

		/**
		 * O'Neil EZ print mode
		 */
		EZ(0),

		/**
		 * O'Neil line printer mode
		 */
		ESC(1),

		/**
		 * O'Neil line printer mode
		 */
		LP(2),

		/**
		 * CPCL Emulation
		 */
		EMZ1(3),

		/**
		 * COG Emulation
		 */
		EMZ2(4),

		/**
		 * ZPL Emulation
		 */
		EMZ3(5),

		/**
		 * ZPL Emulation(shifted 12dots to left)
		 */
		EMZ4(6),

		/**
		 * Custom Emulation
		 */
		EMC1(7),

		/**
		 * Custom Emulation
		 */
		EMC2(8),

		/**
		 * Custom Emulation
		 */
		EMC3(9),

		/**
		 * Custom Emulation
		 */
		EMC4(10),

		/**
		 * PGL Emulation
		 */
		EMP1(11),

		/**
		 * PCL Emulation
		 */
		EMM1(12),

		/**
		 * CPCL Emulation with true fonts
		 */
		EMZ5(14),

		/**
		 * TPCL Emulation
		 */
		EMC6(15);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		EmulationProtocolValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Serial Communication Parity
	 */
	public enum Parity {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * No parity check occurs
		 */
		None(0),

		/**
		 * Sets the parity bit so that the count of bits set is an even number
		 */
		Even(1),

		/**
		 * Sets the parity bit so that the count of bits set is an odd number
		 */
		Odd(2);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		Parity(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * Paper Out Beep
	 */
	public enum PaperOutBeepValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * One Beep
		 */
		OneBeep(0),

		/**
		 * Five beeps
		 */
		FiveBeepsOnce(1),

		/**
		 * Five beeps repeated every 15s
		 */
		FiveBeepsEvery15Sec(2),

		/**
		 * Five beeps repeated every 30s
		 */
		FiveBeepsEvery30Sec(3),

		/**
		 * Five beeps repeated every 60s
		 */
		FiveBeepsEvery60Sec(4);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		PaperOutBeepValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * What USB device class is the printer currently configured for.
	 */
	public enum USBClassType {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * USB CDC class only
		 */
		CDC(0),

		/**
		 * USB Printer class only
		 */
		Printer(1),

		/**
		 * USB CDC as well as Printer class at the same time
		 */
		PrinterCDC(2),

		/**
		 * USB CDC as well as Printer class at the same time using IAD (Interface Association Descriptor)
		 */
		PrinterCDC_IAD(3);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		USBClassType(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * The default constructor will initialize the class with default values.
	 * <p>
	 * Initially all of the _IsValid parameters will be false because no data has been processed.  To 'populate' the values, the object must invoke <see cref="PrinterState.Update(ONeil.Connection.ConnectionBase, int)">Update</see> with the query response string.
	 */
	public GeneralConfiguration() {

		// Set Query Values
		m_QueryDescription = "General Configuration";
		m_Query = "{CF?}";
		m_QueryResponseHeader = "{CF!";

		// Add names
		addName("A", "White Space Advance");
		addName("B", "Baud Rate");
		addName("D", "Darkness Adjustment");
		addName("F", "Form Feed");
		addName("G", "Charger Beep");
		addName("H", "RS232 Handshake");
		addName("I", "Lines Per Page");
		addName("J", "EZ Print Job Status Report");
		addName("L", "Default Protocol");
		addName("LT", "Self Test Print Language");
		addName("M", "Form Feed Centering");
		addName("N", "RS232 Data Bits");
		addName("P", "RS232 Parity");
		addName("PF", "Formfeed Button Disabled");
		addName("PO", "Power Button Disabled");
		addName("PR", "RF Button Disabled");
		addName("QC", "QStop Multiplier");
		addName("R", "RF Power Timeout");
		addName("S", "Sound Enabled");
		addName("T", "System Timeout");
		addName("TP", "Special Test Print");
		addName("U", "Paper Out Beep");
		addName("UC", "USB Class");
		addName("USB", "Using USB");

		return;
	}
	/**
	 * Indicates if getWhiteSpaceAdvance() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getWhiteSpaceAdvance() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getWhiteSpaceAdvance()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getWhiteSpaceAdvance_IsPresent() {
		return containsData("A") && isString("A");
	}

	/**
	 * White Space Advance Configuration
	 * <p>
	 * The return value for this parameter is only valid if getWhiteSpaceAdvance_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getWhiteSpaceAdvance_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getWhiteSpaceAdvance() {
		return parse_boolean("A", "Y", "N");
	}

	/**
	 * Indicates if getBaudRate() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getBaudRate() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getBaudRate()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getBaudRate_IsPresent() {
		return containsData("B") && isString("B");
	}

	/**
	 * Serial Baud Rate
	 * <p>
	 * The return value for this parameter is only valid if getBaudRate_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getBaudRate_IsPresent()
	 * @return The current value for the parameter.
	 */
	public BaudValue getBaudRate() {
		BaudValue result = BaudValue.Unset;

		// Determine value
		if (!containsData("B")) {
			// Value not set
		}
		else if (queryResult("B").equals("012")) {
			// 1200 Baud
			result = BaudValue.Baud_1200;
		}
		else if (queryResult("B").equals("024")) {
			// 2400 Baud
			result = BaudValue.Baud_2400;
		}
		else if (queryResult("B").equals("048")) {
			// 4800 Baud
			result = BaudValue.Baud_4800;
		}
		else if (queryResult("B").equals("096")) {
			// 9600 Baud
			result = BaudValue.Baud_9600;
		}
		else if (queryResult("B").equals("192")) {
			// 19200 Baud
			result = BaudValue.Baud_19200;
		}
		else if (queryResult("B").equals("384")) {
			// 38400 Baud
			result = BaudValue.Baud_38400;
		}
		else if (queryResult("B").equals("576")) {
			// 57600 Baud
			result = BaudValue.Baud_57600;
		}
		else if (queryResult("B").equals("115")) {
			// 115200 Baud
			result = BaudValue.Baud_115200;
		}
		else if (queryResult("B").equals("230")) {
			// 230400 Baud
			result = BaudValue.Baud_230400;
		}
		else if (queryResult("B").equals("460")) {
			// 460800 Baud
			result = BaudValue.Baud_460800;
		}
		else if (queryResult("B").equals("921")) {
			// 921600 Baud
			result = BaudValue.Baud_921600;
		}
		else {
			// Value not yet in DB
			result = BaudValue.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getDarknessAdjustment() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getDarknessAdjustment() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getDarknessAdjustment()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getDarknessAdjustment_IsPresent() {
		return containsData("D") && isInteger("D");
	}

	/**
	 * Printing Darkness Adjustment
	 * <p>
	 * The return value for this parameter is only valid if getDarknessAdjustment_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getDarknessAdjustment_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getDarknessAdjustment() {
		return parse_long("D");
	}

	/**
	 * Indicates if getFormFeed() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getFormFeed() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getFormFeed()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getFormFeed_IsPresent() {
		return containsData("F") && isString("F");
	}

	/**
	 * Form Feed Enabled
	 * <p>
	 * The return value for this parameter is only valid if getFormFeed_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getFormFeed_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getFormFeed() {
		return parse_boolean("F", "Y", "N");
	}

	/**
	 * Indicates if getChargerBeep() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getChargerBeep() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getChargerBeep()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getChargerBeep_IsPresent() {
		return containsData("G") && isString("G");
	}

	/**
	 * Beep When Charger Connected
	 * <p>
	 * The return value for this parameter is only valid if getChargerBeep_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getChargerBeep_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getChargerBeep() {
		return parse_boolean("G", "Y", "N");
	}

	/**
	 * Indicates if getRS232Handshake() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRS232Handshake() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRS232Handshake()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRS232Handshake_IsPresent() {
		return containsData("H") && isString("H");
	}

	/**
	 * Handshake type
	 * <p>
	 * The return value for this parameter is only valid if getRS232Handshake_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRS232Handshake_IsPresent()
	 * @return The current value for the parameter.
	 */
	public Handshake getRS232Handshake() {
		Handshake result = Handshake.Unset;

		// Determine value
		if (!containsData("H")) {
			// Value not set
		}
		else if (queryResult("H").equals("N")) {
			// No control is used for the handshake
			result = Handshake.None;
		}
		else if (queryResult("H").equals("H")) {
			// Request-to-Send (RTS) hardware flow control is used. RTS signals that data is available for transmission. If the input buffer becomes full, the RTS line will be set to false. The RTS line will be set to true when more room becomes available in the input buffer
			result = Handshake.Hardware;
		}
		else if (queryResult("H").equals("S")) {
			// The XON/XOFF software control protocol is used. The XOFF control is sent to stop the transmission of data. The XON control is sent to resume the transmission. These software controls are used instead of Request to Send (RTS) and Clear to Send (CTS) hardware controls
			result = Handshake.Software;
		}
		else if (queryResult("H").equals("B")) {
			// Both the Request-to-Send (RTS) hardware control and the XON/XOFF software controls are used
			result = Handshake.Both;
		}
		else {
			// Value not yet in DB
			result = Handshake.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getLinesPerPage() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getLinesPerPage() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getLinesPerPage()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getLinesPerPage_IsPresent() {
		return containsData("I") && isInteger("I");
	}

	/**
	 * Lines Per Page
	 * <p>
	 * The return value for this parameter is only valid if getLinesPerPage_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getLinesPerPage_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getLinesPerPage() {
		return parse_long("I");
	}

	/**
	 * Indicates if getEZPrintJobStatusReport() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getEZPrintJobStatusReport() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getEZPrintJobStatusReport()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getEZPrintJobStatusReport_IsPresent() {
		return containsData("J") && isString("J");
	}

	/**
	 * Job Status Report
	 * <p>
	 * The return value for this parameter is only valid if getEZPrintJobStatusReport_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getEZPrintJobStatusReport_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getEZPrintJobStatusReport() {
		return parse_boolean("J", "Y", "N");
	}

	/**
	 * Indicates if getDefaultProtocol() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getDefaultProtocol() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getDefaultProtocol()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getDefaultProtocol_IsPresent() {
		return containsData("L") && isString("L");
	}

	/**
	 * Default Printing Protocol
	 * <p>
	 * The return value for this parameter is only valid if getDefaultProtocol_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getDefaultProtocol_IsPresent()
	 * @return The current value for the parameter.
	 */
	public EmulationProtocolValue getDefaultProtocol() {
		EmulationProtocolValue result = EmulationProtocolValue.Unset;

		// Determine value
		if (!containsData("L")) {
			// Value not set
		}
		else if (queryResult("L").equals("PK80")) {
			// PK80 Impact Printer Emulation
			result = EmulationProtocolValue.PK80;
		}
		else if (queryResult("L").equals("EZ")) {
			// O'Neil EZ print mode
			result = EmulationProtocolValue.EZ;
		}
		else if (queryResult("L").equals("ESC")) {
			// O'Neil line printer mode
			result = EmulationProtocolValue.ESC;
		}
		else if (queryResult("L").equals("LP")) {
			// O'Neil line printer mode
			result = EmulationProtocolValue.LP;
		}
		else if (queryResult("L").equals("EMZ1")) {
			// CPCL Emulation
			result = EmulationProtocolValue.EMZ1;
		}
		else if (queryResult("L").equals("EMZ2")) {
			// COG Emulation
			result = EmulationProtocolValue.EMZ2;
		}
		else if (queryResult("L").equals("EMZ3")) {
			// ZPL Emulation
			result = EmulationProtocolValue.EMZ3;
		}
		else if (queryResult("L").equals("EMZ4")) {
			// ZPL Emulation(shifted 12dots to left)
			result = EmulationProtocolValue.EMZ4;
		}
		else if (queryResult("L").equals("EMC1")) {
			// Custom Emulation
			result = EmulationProtocolValue.EMC1;
		}
		else if (queryResult("L").equals("EMC2")) {
			// Custom Emulation
			result = EmulationProtocolValue.EMC2;
		}
		else if (queryResult("L").equals("EMC3")) {
			// Custom Emulation
			result = EmulationProtocolValue.EMC3;
		}
		else if (queryResult("L").equals("EMC4")) {
			// Custom Emulation
			result = EmulationProtocolValue.EMC4;
		}
		else if (queryResult("L").equals("EMP1")) {
			// PGL Emulation
			result = EmulationProtocolValue.EMP1;
		}
		else if (queryResult("L").equals("EMM1")) {
			// PCL Emulation
			result = EmulationProtocolValue.EMM1;
		}
		else if (queryResult("L").equals("EMZ5")) {
			// CPCL Emulation with true fonts
			result = EmulationProtocolValue.EMZ5;
		}
		else if (queryResult("L").equals("EMC6")) {
			// TPCL Emulation
			result = EmulationProtocolValue.EMC6;
		}
		else {
			// Value not yet in DB
			result = EmulationProtocolValue.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getSelfTestPrintLanguage() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getSelfTestPrintLanguage() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getSelfTestPrintLanguage()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getSelfTestPrintLanguage_IsPresent() {
		return containsData("LT") && isInteger("LT");
	}

	/**
	 * Self Test Print Language
	 * <p>
	 * The return value for this parameter is only valid if getSelfTestPrintLanguage_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getSelfTestPrintLanguage_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getSelfTestPrintLanguage() {
		return parse_long("LT");
	}

	/**
	 * Indicates if getFormFeedCentering() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getFormFeedCentering() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getFormFeedCentering()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getFormFeedCentering_IsPresent() {
		return containsData("M") && isString("M");
	}

	/**
	 * Form Feed Centering
	 * <p>
	 * The return value for this parameter is only valid if getFormFeedCentering_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getFormFeedCentering_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getFormFeedCentering() {
		return parse_boolean("M", "Y", "N");
	}

	/**
	 * Indicates if getRS232DataBits() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRS232DataBits() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRS232DataBits()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRS232DataBits_IsPresent() {
		return containsData("N") && isInteger("N");
	}

	/**
	 * Data Bits
	 * <p>
	 * The return value for this parameter is only valid if getRS232DataBits_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRS232DataBits_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getRS232DataBits() {
		return parse_long("N");
	}

	/**
	 * Indicates if getRS232Parity() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRS232Parity() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRS232Parity()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRS232Parity_IsPresent() {
		return containsData("P") && isString("P");
	}

	/**
	 * Serial Communication Parity
	 * <p>
	 * The return value for this parameter is only valid if getRS232Parity_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRS232Parity_IsPresent()
	 * @return The current value for the parameter.
	 */
	public Parity getRS232Parity() {
		Parity result = Parity.Unset;

		// Determine value
		if (!containsData("P")) {
			// Value not set
		}
		else if (queryResult("P").equals("N")) {
			// No parity check occurs
			result = Parity.None;
		}
		else if (queryResult("P").equals("E")) {
			// Sets the parity bit so that the count of bits set is an even number
			result = Parity.Even;
		}
		else if (queryResult("P").equals("O")) {
			// Sets the parity bit so that the count of bits set is an odd number
			result = Parity.Odd;
		}
		else {
			// Value not yet in DB
			result = Parity.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getFormfeedButtonDisabled() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getFormfeedButtonDisabled() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getFormfeedButtonDisabled()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getFormfeedButtonDisabled_IsPresent() {
		return containsData("PF") && isString("PF");
	}

	/**
	 * Formfeed Button Disabled
	 * <p>
	 * The return value for this parameter is only valid if getFormfeedButtonDisabled_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getFormfeedButtonDisabled_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getFormfeedButtonDisabled() {
		return parse_boolean("PF", "Y", "N");
	}

	/**
	 * Indicates if getPowerButtonDisabled() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPowerButtonDisabled() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getPowerButtonDisabled()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPowerButtonDisabled_IsPresent() {
		return containsData("PO") && isString("PO");
	}

	/**
	 * Power Button Disabled
	 * <p>
	 * The return value for this parameter is only valid if getPowerButtonDisabled_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getPowerButtonDisabled_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getPowerButtonDisabled() {
		return parse_boolean("PO", "Y", "N");
	}

	/**
	 * Indicates if getRFButtonDisabled() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRFButtonDisabled() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRFButtonDisabled()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRFButtonDisabled_IsPresent() {
		return containsData("PR") && isString("PR");
	}

	/**
	 * RF Button Disabled
	 * <p>
	 * The return value for this parameter is only valid if getRFButtonDisabled_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRFButtonDisabled_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getRFButtonDisabled() {
		return parse_boolean("PR", "Y", "N");
	}

	/**
	 * Indicates if getQStopMultiplier() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getQStopMultiplier() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getQStopMultiplier()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getQStopMultiplier_IsPresent() {
		return containsData("QC") && isInteger("QC");
	}

	/**
	 * Number of times to add .25 volts or 0x19 to the QMark volt threshold.  The default is 0 and as this number increases the QMark is more difficult to detect, as it decreases the QMark is easier to detect.
	 * <p>
	 * The return value for this parameter is only valid if getQStopMultiplier_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getQStopMultiplier_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getQStopMultiplier() {
		return parse_long("QC");
	}

	/**
	 * Indicates if getRFPowerTimeout() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getRFPowerTimeout() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRFPowerTimeout()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getRFPowerTimeout_IsPresent() {
		return containsData("R") && isInteger("R");
	}

	/**
	 * RF Power Timeout
	 * <p>
	 * The return value for this parameter is only valid if getRFPowerTimeout_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getRFPowerTimeout_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getRFPowerTimeout() {
		return parse_long("R");
	}

	/**
	 * Indicates if getSoundEnabled() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getSoundEnabled() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getSoundEnabled()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getSoundEnabled_IsPresent() {
		return containsData("S") && isString("S");
	}

	/**
	 * Sound Enabled
	 * <p>
	 * The return value for this parameter is only valid if getSoundEnabled_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getSoundEnabled_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getSoundEnabled() {
		return parse_boolean("S", "Y", "N");
	}

	/**
	 * Indicates if getSystemTimeout() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getSystemTimeout() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getSystemTimeout()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getSystemTimeout_IsPresent() {
		return containsData("T") && isString("T");
	}

	/**
	 * System Timeout
	 * <p>
	 * The return value for this parameter is only valid if getSystemTimeout_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getSystemTimeout_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getSystemTimeout() {
		return parse_string("T");
	}

	/**
	 * Indicates if getSpecialTestPrint() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getSpecialTestPrint() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getSpecialTestPrint()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getSpecialTestPrint_IsPresent() {
		return containsData("TP") && isInteger("TP");
	}

	/**
	 * Special Test Print Mode
	 * <p>
	 * The return value for this parameter is only valid if getSpecialTestPrint_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getSpecialTestPrint_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getSpecialTestPrint() {
		return parse_long("TP");
	}

	/**
	 * Indicates if getPaperOutBeep() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPaperOutBeep() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getPaperOutBeep()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPaperOutBeep_IsPresent() {
		return containsData("U") && isString("U");
	}

	/**
	 * Paper Out Beep
	 * <p>
	 * The return value for this parameter is only valid if getPaperOutBeep_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getPaperOutBeep_IsPresent()
	 * @return The current value for the parameter.
	 */
	public PaperOutBeepValue getPaperOutBeep() {
		PaperOutBeepValue result = PaperOutBeepValue.Unset;

		// Determine value
		if (!containsData("U")) {
			// Value not set
		}
		else if (queryResult("U").equals("0")) {
			// One Beep
			result = PaperOutBeepValue.OneBeep;
		}
		else if (queryResult("U").equals("1")) {
			// Five beeps
			result = PaperOutBeepValue.FiveBeepsOnce;
		}
		else if (queryResult("U").equals("2")) {
			// Five beeps repeated every 15s
			result = PaperOutBeepValue.FiveBeepsEvery15Sec;
		}
		else if (queryResult("U").equals("3")) {
			// Five beeps repeated every 30s
			result = PaperOutBeepValue.FiveBeepsEvery30Sec;
		}
		else if (queryResult("U").equals("4")) {
			// Five beeps repeated every 60s
			result = PaperOutBeepValue.FiveBeepsEvery60Sec;
		}
		else {
			// Value not yet in DB
			result = PaperOutBeepValue.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getUSBClass() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getUSBClass() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getUSBClass()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getUSBClass_IsPresent() {
		return containsData("UC") && isString("UC");
	}

	/**
	 * What USB device class is the printer currently configured for.
	 * <p>
	 * The return value for this parameter is only valid if getUSBClass_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getUSBClass_IsPresent()
	 * @return The current value for the parameter.
	 */
	public USBClassType getUSBClass() {
		USBClassType result = USBClassType.Unset;

		// Determine value
		if (!containsData("UC")) {
			// Value not set
		}
		else if (queryResult("UC").equals("0")) {
			// USB CDC class only
			result = USBClassType.CDC;
		}
		else if (queryResult("UC").equals("1")) {
			// USB Printer class only
			result = USBClassType.Printer;
		}
		else if (queryResult("UC").equals("2")) {
			// USB CDC as well as Printer class at the same time
			result = USBClassType.PrinterCDC;
		}
		else if (queryResult("UC").equals("3")) {
			// USB CDC as well as Printer class at the same time using IAD (Interface Association Descriptor)
			result = USBClassType.PrinterCDC_IAD;
		}
		else {
			// Value not yet in DB
			result = USBClassType.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getUsingUSB() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getUsingUSB() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getUsingUSB()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getUsingUSB_IsPresent() {
		return containsData("USB") && isString("USB");
	}

	/**
	 * Is USB the current communication method.
	 * <p>
	 * The return value for this parameter is only valid if getUsingUSB_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see GeneralConfiguration#getUsingUSB_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getUsingUSB() {
		return parse_boolean("USB", "1", "0");
	}

}
