package datamaxoneil.printer.configuration;

/**
 * This class handles the data from an {BT?} query as well as
 * accessors to read the data values.
 * <p>
 * This class supports the following possible parameters:
 * <table border=1>
 *    <tr><th>Parameter</th><th>Description</th></tr>
 *    <tr><td>CH</td><td>Is the printer plugged into an AC power source</td></tr>
 *    <tr><td>PS</td><td>Power source</td></tr>
 *    <tr><td>T</td><td>Temperature of the battery</td></tr>
 *    <tr><td>V</td><td>Voltage of battery</td></tr>
 *    <tr><td>V1</td><td>Voltage of battery 1</td></tr>
 *    <tr><td>V2</td><td>Voltage of battery 2</td></tr>
 *    <tr><td>VE</td><td>Voltage of the battery eliminator</td></tr>
 * </table>
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class BatteryCondition extends PrinterState {

	/**
	 * Power source
	 */
	public enum PowerSourceValue {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * DC power is configured to use in preference over the battery.
		 */
		DCOverBattery(65),

		/**
		 * Battery is configured to use in preference over the DC power.
		 */
		BatteryOverDC(66),

		/**
		 * The battery eliminator is being used.
		 */
		BatteryEliminator(67);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		PowerSourceValue(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * The default constructor will initialize the class with default values.
	 * <p>
	 * Initially all of the _IsValid parameters will be false because no data has been processed.  To 'populate' the values, the object must invoke <see cref="PrinterState.Update(ONeil.Connection.ConnectionBase, int)">Update</see> with the query response string.
	 */
	public BatteryCondition() {

		// Set Query Values
		m_QueryDescription = "Battery Condition";
		m_Query = "{BT?}";
		m_QueryResponseHeader = "{BT!";

		// Add names
		addName("CH", "Charger Connected");
		addName("PS", "Power Source");
		addName("T", "Battery Temperature");
		addName("V", "Voltage Battery Single");
		addName("V1", "Voltage Battery 1");
		addName("V2", "Voltage Battery 2");
		addName("VE", "Voltage Battery Eliminator");

		return;
	}
	/**
	 * Indicates if getChargerConnected() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getChargerConnected() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getChargerConnected()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getChargerConnected_IsPresent() {
		return containsData("CH") && isString("CH");
	}

	/**
	 * Is the printer plugged into an AC power source
	 * <p>
	 * The return value for this parameter is only valid if getChargerConnected_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getChargerConnected_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getChargerConnected() {
		return parse_boolean("CH", "Y", "N");
	}

	/**
	 * Indicates if getPowerSource() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPowerSource() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getPowerSource()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPowerSource_IsPresent() {
		return containsData("PS") && isString("PS");
	}

	/**
	 * Power source
	 * <p>
	 * The return value for this parameter is only valid if getPowerSource_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getPowerSource_IsPresent()
	 * @return The current value for the parameter.
	 */
	public PowerSourceValue getPowerSource() {
		PowerSourceValue result = PowerSourceValue.Unset;

		// Determine value
		if (!containsData("PS")) {
			// Value not set
		}
		else if (queryResult("PS").equals("A")) {
			// DC power is configured to use in preference over the battery.
			result = PowerSourceValue.DCOverBattery;
		}
		else if (queryResult("PS").equals("B")) {
			// Battery is configured to use in preference over the DC power.
			result = PowerSourceValue.BatteryOverDC;
		}
		else if (queryResult("PS").equals("C")) {
			// The battery eliminator is being used.
			result = PowerSourceValue.BatteryEliminator;
		}
		else {
			// Value not yet in DB
			result = PowerSourceValue.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getBatteryTemperature() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getBatteryTemperature() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getBatteryTemperature()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getBatteryTemperature_IsPresent() {
		return containsData("T") && isDouble("T");
	}

	/**
	 * Temperature of the battery
	 * <p>
	 * The return value for this parameter is only valid if getBatteryTemperature_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getBatteryTemperature_IsPresent()
	 * @return The current value for the parameter.
	 */
	public double getBatteryTemperature() {
		return parse_double("T");
	}

	/**
	 * Indicates if getVoltageBatterySingle() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getVoltageBatterySingle() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getVoltageBatterySingle()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getVoltageBatterySingle_IsPresent() {
		return containsData("V") && isDouble("V");
	}

	/**
	 * Voltage of battery
	 * <p>
	 * The return value for this parameter is only valid if getVoltageBatterySingle_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getVoltageBatterySingle_IsPresent()
	 * @return The current value for the parameter.
	 */
	public double getVoltageBatterySingle() {
		return parse_double("V");
	}

	/**
	 * Indicates if getVoltageBattery1() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getVoltageBattery1() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getVoltageBattery1()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getVoltageBattery1_IsPresent() {
		return containsData("V1") && isDouble("V1");
	}

	/**
	 * Voltage of battery 1
	 * <p>
	 * The return value for this parameter is only valid if getVoltageBattery1_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getVoltageBattery1_IsPresent()
	 * @return The current value for the parameter.
	 */
	public double getVoltageBattery1() {
		return parse_double("V1");
	}

	/**
	 * Indicates if getVoltageBattery2() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getVoltageBattery2() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getVoltageBattery2()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getVoltageBattery2_IsPresent() {
		return containsData("V2") && isDouble("V2");
	}

	/**
	 * Voltage of battery 2
	 * <p>
	 * The return value for this parameter is only valid if getVoltageBattery2_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getVoltageBattery2_IsPresent()
	 * @return The current value for the parameter.
	 */
	public double getVoltageBattery2() {
		return parse_double("V2");
	}

	/**
	 * Indicates if getVoltageBatteryEliminator() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getVoltageBatteryEliminator() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getVoltageBatteryEliminator()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getVoltageBatteryEliminator_IsPresent() {
		return containsData("VE") && isDouble("VE");
	}

	/**
	 * Voltage of the battery eliminator
	 * <p>
	 * The return value for this parameter is only valid if getVoltageBatteryEliminator_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see BatteryCondition#getVoltageBatteryEliminator_IsPresent()
	 * @return The current value for the parameter.
	 */
	public double getVoltageBatteryEliminator() {
		return parse_double("VE");
	}

}
