package datamaxoneil.printer.configuration;

/**
 * This class handles the data from an {AE?} query as well as
 * accessors to read the data values.
 * <p>
 * This class supports the following possible parameters:
 * <table border=1>
 *    <tr><th>Parameter</th><th>Description</th></tr>
 *    <tr><td>AIP</td><td>What is the IP address of the Avalanche agent</td></tr>
 *    <tr><td>ALL</td><td>Should all data be printed durning a self test.</td></tr>
 *    <tr><td>ANA</td><td>What is the name of the Avalanche agent</td></tr>
 *    <tr><td>APT</td><td>What is the port number of the Avalanche agent</td></tr>
 *    <tr><td>CON</td><td>Type of connection being used to talk to the printer.</td></tr>
 *    <tr><td>MLN</td><td>What is the printer's model name</td></tr>
 *    <tr><td>PRE</td><td>Is a prelicensed block being used</td></tr>
 *    <tr><td>PRN</td><td>Printers TCPIP station name</td></tr>
 *    <tr><td>PRS</td><td>When an Avalanche update is performed this will print out the results if true.</td></tr>
 *    <tr><td>RUN</td><td>Is Avalanche support enabled on the printer.</td></tr>
 *    <tr><td>TTO</td><td>TCP timeout in milliseconds</td></tr>
 *    <tr><td>UIN</td><td>Update interval in minutes</td></tr>
 *    <tr><td>UPD</td><td>Flag is used to tell us how/when the update will be occur</td></tr>
 *    <tr><td>UTO</td><td>UDP timeout in milliseconds</td></tr>
 *    <tr><td>WRD</td><td>Are we using a wired connection</td></tr>
 * </table>
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class AvalancheSettings extends PrinterState {

	/**
	 * Type of connection being used to talk to the printer.
	 */
	public enum Connection {
		/**
		 * The result was never set.
		 */
		Unset(-2),

		/**
		 * The result was unrecognized.  This usually results from a new value 
		 * since the release of the SDK.
		 */
		Unknown(-1),

		/**
		 * Serial Connection
		 */
		Serial(72),

		/**
		 * TCP or UDP Connection
		 */
		TCP_UDP(70);

		/** Numeric Value */
		private int m_Value;

		/** Initialized Constructor
		 *  @param value Numeric value for the enum
		 */
		Connection(int value) {
			m_Value = value;
			return;
		}

		/** Retrieve Item Value
		 *  @return Numeric value for the enum
		 */
		public int value() { return m_Value;}
	}

	/**
	 * The default constructor will initialize the class with default values.
	 * <p>
	 * Initially all of the _IsValid parameters will be false because no data has been processed.  To 'populate' the values, the object must invoke <see cref="PrinterState.Update(ONeil.Connection.ConnectionBase, int)">Update</see> with the query response string.
	 */
	public AvalancheSettings() {

		// Set Query Values
		m_QueryDescription = "Avalanche Settings";
		m_Query = "{AE?}";
		m_QueryResponseHeader = "{AE!";

		// Add names
		addName("AIP", "Agent IP");
		addName("ALL", "Selftest Data");
		addName("ANA", "Agent Name");
		addName("APT", "Agent Port");
		addName("CON", "Connection Type");
		addName("MLN", "Printer Model Name");
		addName("PRE", "Prelicensed");
		addName("PRN", "Printer Name");
		addName("PRS", "Printer Result Flag");
		addName("RUN", "Avalanche Enabled");
		addName("TTO", "TCP Connect Timeout");
		addName("UIN", "Update Interval");
		addName("UPD", "Update Flags");
		addName("UTO", "UDP Timeout");
		addName("WRD", "Wired");

		return;
	}
	/**
	 * Indicates if getAgentIP() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getAgentIP() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getAgentIP()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getAgentIP_IsPresent() {
		return containsData("AIP") && isString("AIP");
	}

	/**
	 * What is the IP address of the Avalanche agent
	 * <p>
	 * The return value for this parameter is only valid if getAgentIP_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getAgentIP_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getAgentIP() {
		return parse_string("AIP");
	}

	/**
	 * Indicates if getShowAllData() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getShowAllData() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getShowAllData()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getShowAllData_IsPresent() {
		return containsData("ALL") && isString("ALL");
	}

	/**
	 * Should all data be printed durning a self test.
	 * <p>
	 * The return value for this parameter is only valid if getShowAllData_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getShowAllData_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getShowAllData() {
		return parse_boolean("ALL", "y", "n");
	}

	/**
	 * Indicates if getAgentName() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getAgentName() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getAgentName()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getAgentName_IsPresent() {
		return containsData("ANA") && isString("ANA");
	}

	/**
	 * What is the name of the Avalanche agent
	 * <p>
	 * The return value for this parameter is only valid if getAgentName_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getAgentName_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getAgentName() {
		return parse_string("ANA");
	}

	/**
	 * Indicates if getAgentPort() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getAgentPort() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getAgentPort()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getAgentPort_IsPresent() {
		return containsData("APT") && isInteger("APT");
	}

	/**
	 * What is the port number of the Avalanche agent
	 * <p>
	 * The return value for this parameter is only valid if getAgentPort_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getAgentPort_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getAgentPort() {
		return parse_long("APT");
	}

	/**
	 * Indicates if getConnectionType() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getConnectionType() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getConnectionType()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getConnectionType_IsPresent() {
		return containsData("CON") && isString("CON");
	}

	/**
	 * Type of connection being used to talk to the printer.
	 * <p>
	 * The return value for this parameter is only valid if getConnectionType_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getConnectionType_IsPresent()
	 * @return The current value for the parameter.
	 */
	public Connection getConnectionType() {
		Connection result = Connection.Unset;

		// Determine value
		if (!containsData("CON")) {
			// Value not set
		}
		else if (queryResult("CON").equals("r")) {
			// Serial Connection
			result = Connection.Serial;
		}
		else if (queryResult("CON").equals("p")) {
			// TCP or UDP Connection
			result = Connection.TCP_UDP;
		}
		else {
			// Value not yet in DB
			result = Connection.Unknown;
		}

		return result; 
	}

	/**
	 * Indicates if getPrinterModelName() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPrinterModelName() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getPrinterModelName()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPrinterModelName_IsPresent() {
		return containsData("MLN") && isString("MLN");
	}

	/**
	 * What is the printer's model name
	 * <p>
	 * The return value for this parameter is only valid if getPrinterModelName_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getPrinterModelName_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getPrinterModelName() {
		return parse_string("MLN");
	}

	/**
	 * Indicates if getIsPrelicensed() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getIsPrelicensed() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getIsPrelicensed()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getIsPrelicensed_IsPresent() {
		return containsData("PRE") && isString("PRE");
	}

	/**
	 * Is a prelicensed block being used
	 * <p>
	 * The return value for this parameter is only valid if getIsPrelicensed_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getIsPrelicensed_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getIsPrelicensed() {
		return parse_boolean("PRE", "1", "0");
	}

	/**
	 * Indicates if getPrinterName() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPrinterName() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getPrinterName()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPrinterName_IsPresent() {
		return containsData("PRN") && isString("PRN");
	}

	/**
	 * Printers TCPIP station name
	 * <p>
	 * The return value for this parameter is only valid if getPrinterName_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getPrinterName_IsPresent()
	 * @return The current value for the parameter.
	 */
	public String getPrinterName() {
		return parse_string("PRN");
	}

	/**
	 * Indicates if getPrinterResultFlag() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getPrinterResultFlag() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getPrinterResultFlag()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getPrinterResultFlag_IsPresent() {
		return containsData("PRS") && isString("PRS");
	}

	/**
	 * When an Avalanche update is performed this will print out the results if true.
	 * <p>
	 * The return value for this parameter is only valid if getPrinterResultFlag_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getPrinterResultFlag_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getPrinterResultFlag() {
		return parse_boolean("PRS", "y", "n");
	}

	/**
	 * Indicates if getIsAvalancheEnabled() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getIsAvalancheEnabled() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getIsAvalancheEnabled()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getIsAvalancheEnabled_IsPresent() {
		return containsData("RUN") && isString("RUN");
	}

	/**
	 * Is Avalanche support enabled on the printer.
	 * <p>
	 * The return value for this parameter is only valid if getIsAvalancheEnabled_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getIsAvalancheEnabled_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getIsAvalancheEnabled() {
		return parse_boolean("RUN", "1", "0");
	}

	/**
	 * Indicates if getTCP_ConnectTimeout() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getTCP_ConnectTimeout() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getTCP_ConnectTimeout()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getTCP_ConnectTimeout_IsPresent() {
		return containsData("TTO") && isInteger("TTO");
	}

	/**
	 * TCP timeout in milliseconds
	 * <p>
	 * The return value for this parameter is only valid if getTCP_ConnectTimeout_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getTCP_ConnectTimeout_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getTCP_ConnectTimeout() {
		return parse_long("TTO");
	}

	/**
	 * Indicates if getUpdateInterval() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getUpdateInterval() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getUpdateInterval()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getUpdateInterval_IsPresent() {
		return containsData("UIN") && isInteger("UIN");
	}

	/**
	 * Update interval in minutes
	 * <p>
	 * The return value for this parameter is only valid if getUpdateInterval_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getUpdateInterval_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getUpdateInterval() {
		return parse_long("UIN");
	}

	/**
	 * Indicates if getUpdateFlags() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getUpdateFlags() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getUpdateFlags()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getUpdateFlags_IsPresent() {
		return containsData("UPD") && isInteger("UPD");
	}

	/**
	 * Flag is used to tell us how/when the update will be occur
	 * <p>
	 * The return value for this parameter is only valid if getUpdateFlags_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getUpdateFlags_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getUpdateFlags() {
		return parse_long("UPD");
	}

	/**
	 * Indicates if getUDP_Timeout() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getUDP_Timeout() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getUDP_Timeout()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getUDP_Timeout_IsPresent() {
		return containsData("UTO") && isInteger("UTO");
	}

	/**
	 * UDP timeout in milliseconds
	 * <p>
	 * The return value for this parameter is only valid if getUDP_Timeout_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getUDP_Timeout_IsPresent()
	 * @return The current value for the parameter.
	 */
	public long getUDP_Timeout() {
		return parse_long("UTO");
	}

	/**
	 * Indicates if getIsWired() parameter is present.
	 * <p>
	 * This function is useful to determine if the given parameter is supported on the current printer.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned by the parameter getIsWired() may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getIsWired()
	 * @return If the parameter is present and valid true, false otherwise.
	 */
	public boolean getIsWired_IsPresent() {
		return containsData("WRD") && isString("WRD");
	}

	/**
	 * Are we using a wired connection
	 * <p>
	 * The return value for this parameter is only valid if getIsWired_IsPresent() returns true.  Because of differences in configuration, hardware or version certain parameter may not exist and the results returned may not be valid.  Instead of trying to indicate this with 'Magic Values' this method was implemented.
	 *
	 * @see AvalancheSettings#getIsWired_IsPresent()
	 * @return The current value for the parameter.
	 */
	public boolean getIsWired() {
		return parse_boolean("WRD", "1", "0");
	}

}
