package datamaxoneil.printer;


/** 
Class containing UPS message for printing Maxicode Bar Codes
*/
public class UPSMessage
{
	/** 
	 Default Constructor
	*/
	public UPSMessage()
	{

	}
	/** 
	 This constructor takes all the properties of the message specified by user. 
	 
	 @param zipCode 5 or 9 digit zip code(US) or 6 alphanumeric characters zip code(International) 
	 @param countryCode 3-digit Country Code
	 @param classOfService 3-digit class of service
	 @param trackingNum Shipment Tracking Number (10 or 11 uppercase alphanumeric characters)
	 @param scac Standard Carrier Alpha Code(4 uppercase alpha characters)
	 @param shipperNum UPS shipper number (6 uppercase alphanumeric characters)
	 @param pickupDay Julian day of pickup
	 @param shipID Optional. Shipment ID Number (0-30 alpha/numeric characters). If not use, pass in empty string.
	 @param numberOfPackages Total number of packages for shipment(1-3 digits)
	 @param packageNumber Current package number(1-3 digits)
	 @param packageWeight Package weight in pounds(1-3 digits)
	 @param validateAddr Address validation.
	 @param shipToAddr Optional. Ship-to Address (0-35 alpha/numeric characters). If not use, pass in empty string.
	 @param shipToCity Ship-to City (1-20 alpha/numeric characters)
	 @param shipToState Ship-to State(2 alpha characters)
	*/
	public UPSMessage(String zipCode, int countryCode, int classOfService, String trackingNum, String scac, String shipperNum, int pickupDay, String shipID, int numberOfPackages, int packageNumber, int packageWeight, boolean validateAddr, String shipToAddr, String shipToCity, String shipToState)
	{
		setZipCode(zipCode);
		setCountryCode(countryCode);
		setClassOfService(classOfService);
		setTrackingNumber(trackingNum);
		setSCAC(scac);
		setShipperNumber(shipperNum);
		setJulianPickupDay(pickupDay);
		setShipmentID(shipID);
		setTotalPackage(numberOfPackages);
		setCurrentPackage(packageNumber);
		setPackageWeight(packageWeight);
		setValidateAddress(validateAddr);
		setShipToAddr(shipToAddr);
		setShipToCity(shipToCity);
		setShipToState(shipToState);
	}

	/** 
	 5 or 9 alphanumeric post code.
	 US - 5-digit zip code + 4 digit zip extension
	 International - 6 alphanumeric character zip code(A-Z, 0-9)
	*/
	private String m_zipCode;
	/** 
	 5 or 9 alphanumeric post code.
	 US - 5-digit zip code + 4 digit zip extension
	 International - 6 alphanumeric character zip code(A-Z, 0-9)
	*/
	public final String getZipCode()
	{
		return m_zipCode;
	}
	public final void setZipCode(String value)
	{
		if (((value.length() != 5 && value.length() != 9) || !Document.matches(value, "^[0-9]*$")) && ((value.length() != 6) || !Document.matches(value, "^[A-Z0-9]*$")))
		{
			throw new IllegalArgumentException(String.format("Zip code must be 5-9 numeric digits or 6 alphanumeric characters. Value given was %1$s", value));
		}
		m_zipCode = value;
	}

	/** 
	 Ship-to country code. Default is US Country code
	*/
	private int m_countryCode = 840;

	/** 
	 Ship-to country code
	*/
	public final int getCountryCode()
	{
		return m_countryCode;
	}
	public final void setCountryCode(int value)
	{
		if (value < 0 || value > 999)
		{
			throw new IllegalArgumentException(String.format("CountryCode must be between 0 and 999. Value given was %1$s", value));
		}

		m_countryCode = value;
	}
	/** 
	 3-Digit Class Of Service
	*/
	private int m_classOfService = 0;
	/** 
	 3-Digit Class Of Service
	*/
	public final int getClassOfService()
	{
		return m_classOfService;
	}
	public final void setClassOfService(int value)
	{
		if (value < 0 || value > 999)
		{
			throw new IllegalArgumentException(String.format("ClassOfService must be between 0 and 999. Value given was %1$s", value));
		}
		m_classOfService = value;
	}

	/** 
	 Shipment Tracking Number 
	*/
	private String m_trackingNumber = "1Z01234567";
	/** 
	 Shipment Tracking Number 
	*/
	public final String getTrackingNumber()
	{
		return m_trackingNumber;
	}
	public final void setTrackingNumber(String value)
	{
			//Set value if input is 10-11 alphanumeric characters
		 if ((value.length() != 10 && value.length() != 11) || !Document.matches(value,"^[A-Z0-9]*$"))
		 {
			 throw new IllegalArgumentException(String.format("TrackingNumber must be 10 or 11 alphanumeric characters(Alpha must be uppercase). Value given was %1$s.", value));
		 }
		 m_trackingNumber = value;


	}

	/** 
	 Standard Carrier Alpha Code (SCAC)
	*/
	private String m_SCAC = "UPSN";

	/** 
	 Standard Carrier Alpha Code (SCAC)
	*/
	public final String getSCAC()
	{
		return m_SCAC;
	}
	public final void setSCAC(String value)
	{
		if (value.length() != 4 || !Document.matches(value, "^[A-Z]*$"))
		{
			throw new IllegalArgumentException(String.format("SCAC must be 4 uppercase alpha characters. Value given was %1$s.", value));
		}
		m_SCAC = value;
	}

	/** 
	 UPS Shipper Number
	*/
	private String m_shipperNumber = "123456";

	/** 
	 UPS Shipper Number
	*/
	public final String getShipperNumber()
	{
		return m_shipperNumber;
	}
	public final void setShipperNumber(String value)
	{
		if (!Document.matches(value,"[A-Z0-9]{6}"))
		{
			 throw new IllegalArgumentException(String.format("ShipperNumber must be 6 alphanumeric characters(Alpha must be uppercase). Value given was %1$s", value));
		}
		m_shipperNumber = value;
	}

	/** 
	 Julian Day of Pickup
	*/
	private int m_julianPickupDay = 1;

	/** 
	 Julian Day of Pickup
	*/
	public final int getJulianPickupDay()
	{
		return m_julianPickupDay;
	}
	public final void setJulianPickupDay(int value)
	{
		if (value < 1 || value > 365)
		{
			throw new IllegalArgumentException(String.format("JulianPickupDay must be between 1 and 365. Value given was %1$s", value));
		}
		m_julianPickupDay = value;
	}

	/** 
	 Shipment ID Number
	*/
	private String m_shipmentID = "";

	/** 
	 Shipment ID Number.Optional
	*/
	public final String getShipmentID()
	{
		return m_shipmentID;
	}
	public final void setShipmentID(String value)
	{
		 if (value.length() > 30 || !Document.matches(value,"^[A-Z0-9]*$"))
		 {
			 throw new IllegalArgumentException(String.format("ShipmentID  must be 0-30 alphanumeric characters(Alpha must be uppercase). Value given was %1$s", value));
		 }
		m_shipmentID = value;
	}

	/** 
	 Current package number of shipment
	*/
	private int m_currentPackage = 1;
	/** 
	 Current package number of shipment
	*/
	public final int getCurrentPackage()
	{
		return m_currentPackage;
	}
	public final void setCurrentPackage(int value)
	{
		if (value < 0 || value > 999 || value > getTotalPackage())
		{
			throw new IllegalArgumentException(String.format("currentPackage must be between 0-999. Value given was %1$s", value));
		}
			m_currentPackage = value;
	}
	/** 
	 Total packages for shipment
	*/
	private int m_totalPackage = 1;

	/** 
	 Total packages for shipment
	*/
	public final int getTotalPackage()
	{
		return m_totalPackage;
	}
	public final void setTotalPackage(int value)
	{
		if ((value < 0 || value > 999) || value < getCurrentPackage())
		{
			throw new IllegalArgumentException(String.format("totalPackage must be between 0 - 999 OR greater than/equal to the currentPackage number. Value given was %1$s", value));
		}
		m_totalPackage = value;
	}
	/** 
	 Package Weight in pounds (1-3 numeric digits
	*/
	private int m_packageWeight = 1;

	/** 
	 Package Weight in pounds (1-3 numeric digits
	*/
	public final int getPackageWeight()
	{
		return m_packageWeight;
	}
	public final void setPackageWeight(int value)
	{
		if (value < 0 || value > 999)
		{
			throw new IllegalArgumentException(String.format("PackageWeight must be between 0 and 999. Value given was %1$s", value));
		}
		m_packageWeight = value;
	}

	/** 
	 Address Validation
	*/
	private boolean m_validateAddress = true;

	/** 
	 Address Validation
	*/
	public final boolean getValidateAddress()
	{
		return m_validateAddress;
	}
	public final void setValidateAddress(boolean value)
	{
		m_validateAddress = value;
	}

	/** 
	 Ship-to Address. Optional
	*/
	private String m_shipToAddr = "1234 APPLE ST";

	/** 
	 Ship-to Address
	*/
	public final String getShipToAddr()
	{
		return m_shipToAddr;
	}
	public final void setShipToAddr(String value)
	{
		if (value.length() > 35 || !Document.matches(value,"^[A-Z0-9 ]*$"))
		{
			 throw new IllegalArgumentException(String.format("ShipToAddr  must be 0-35 alphanumeric characters(Alpha must be uppercase). Value given was %1$s", value));
		}
		m_shipToAddr = value;
	}

	/** 
	 Ship-To City
	*/
	private String m_shipToCity = "ORANGE";

	/** 
	 Ship-To City
	*/
	public final String getShipToCity()
	{
		return m_shipToCity;
	}
	public final void setShipToCity(String value)
	{
		if ((value.length() < 1 || value.length() >20) || !Document.matches(value,"^[A-Z0-9 ]*$"))
		{
			throw new IllegalArgumentException(String.format("ShipToAddr  must be 1-20 alphanumeric characters(Alpha must be uppercase). Value given was %1$s", value));
		}
		m_shipToCity = value;
	}
	/** 
	 Ship-To State
	*/
	private String m_shipToState = "CA";

	/** 
	 Ship-To State
	*/
	public final String getShipToState()
	{
		return m_shipToState;
	}
	public final void setShipToState(String value)
	{
		if (value.length() != 2 || !Document.matches(value,"^[A-Z]*$"))
		{
			 throw new IllegalArgumentException(String.format("ShipToAddr  must be 2 uppercase alpha characters. Value given was %1$s", value));
		}
		m_shipToState = value;
	}


}