package datamaxoneil.printer;

import datamaxoneil.printer.Parameters;


/**
 * This class represents the general parameters that can be used to alter the
 * printing characteristics when 'printing' to a DocumentEZ object. Printing
 * characteristics are 'local' to each print command they are passed into. To
 * have global settings the same parameters object can be passed into each write
 * call to the Document object.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class ParametersEZ extends Parameters {

	/** Amount to rotate the printing. */
	public enum Rotation {
		/** Value for rotating the printing by 0 degrees, ie no rotation. */
		Rotate_0,
		/** Value for rotating the printing by 90 degrees counterclockwise. */
		Rotate_90,
		/** Value for rotating the printing by 180 degrees counterclockwise. */
		Rotate_180,
		/** Value for rotating the printing by 270 degrees counterclockwise. */
		Rotate_270
	}

	/** Alignment of the printing object */
	public enum Alignment {
		/** Value for Left alignment of a print field. */
		Left,	
		/** Value for Center alignment of a print field. */
		Center,	
		/** Value for Right alignment of a print field. */
		Right
	}
	
	/** Degrees to rotate the printing. */
	private Rotation m_Rotation = Rotation.Rotate_0;

	/** Degrees to rotate the printing. */
	private Alignment m_Alignment = Alignment.Left;

	/**
	 * This is the number of degrees the printing will be rotated about the
	 * starting location in the counter-clockwise direction. The default value
	 * for this parameter is 0 degrees.
	 * 
	 * @return The number of degrees the printing will be rotated
	 *         counter-clockwise.
	 * @since 1.1.0
	 */
	public Rotation getRotate() {
		return m_Rotation;
	}

	/**
	 * This is the horizontal alignment to use when writing the item. The
	 * options are:
	 * <ul>
	 * <li>Alignment.Left - Left aligned. This will write the item starting at
	 * the column specified in the write command.</li>
	 * <li>Alignment.Center - Center aligned. This will write the item centered
	 * at the column specified in the write command.</li>
	 * <li>Alignment.Right - Right aligned. This will write the item flush
	 * against the right side of the column specified in the write command.</li>
	 * </ul>
	 * 
	 * @return Alignment setting to use.
	 */
	public Alignment getAlignment() {
		return m_Alignment;
	}

	/**
	 * The font name is a five character string that represents the name of the
	 * font, already loaded on the printer, that you wish to print with.
	 * 
	 * @param fontName The five character font name or the empty string if the
	 *        default font is to be used.
	 * @since 1.0.0
	 */
	public void setFont(String fontName) {

		// Assign Value
		setFont(fontName, 5);

		return;
	}

	/**
	 * The horizontal multiplier property will cause the horizontal printing of
	 * the object to be scaled by that amount. So a value of two would cause a
	 * printing of the object to be double its original width. The default value
	 * for this parameter is 1 and the range of valid values is 1 to 255.
	 * 
	 * @param multiplier Factor by which to scale the printing horizontally.
	 * @throws IllegalArgumentException If the parameter values are not in the
	 *         valid range.
	 * @since 1.0.0
	 */
	public void setHorizontalMultiplier(int multiplier) throws IllegalArgumentException {

		// Assign Value
		setHorizontalMultiplier(multiplier, 1, 255);

		return;
	}

	/**
	 * The vertical multiplier property will cause the vertical printing of the
	 * object to be scaled by that amount. So a value of two would cause a
	 * printing of the object to be double its original hieght. The default
	 * value for this parameter is 1 and the range of valid values is 1 to 255.
	 * 
	 * @param multiplier Factor by which to scale the printing vertically.
	 * @throws IllegalArgumentException If the parameter values are not in the
	 *         valid range.
	 * @since 1.0.0
	 */
	public void setVerticalMultiplier(int multiplier) throws IllegalArgumentException {

		// Assign Value
		setVerticalMultiplier(multiplier, 1, 255);

		return;
	}

	/**
	 * This is the number of degrees the printing will be rotated about the
	 * starting location in the counter-clockwise direction. The default value
	 * for this parameter is 0 degrees.
	 * 
	 * @param rotationAngle Number of degrees to rotate the printing the range
	 *        of valid values is 0, 90, 180 or 270 degrees.
	 * @throws IllegalArgumentException If the parameter values are not in the
	 *         valid range.
	 * @since 1.1.0
	 */
	public void setRotate(Rotation rotationAngle) throws IllegalArgumentException {

		// Check inputs
		if (rotationAngle == null) {
			throw new IllegalArgumentException("Parameter 'rotationAngle' cannot be null.");		
		}

		// Assign Value
		m_Rotation = rotationAngle;

		return;
	}

	/**
	 * This is the horizontal alignment to use when writing the item. The
	 * options are:
	 * <ul>
	 * <li>Alignment.Left - Left aligned. This will write the item starting at
	 * the column specified in the write command.</li>
	 * <li>Alignment.Center - Center aligned. This will write the item centered
	 * at the column specified in the write command.</li>
	 * <li>Alignment.Right - Right aligned. This will write the item flush
	 * against the right side of the column specified in the write command.</li>
	 * </ul>
	 * 
	 * @param alignment Alignment setting to use.
	 */
	public void setAlignment(Alignment alignment) {

		// Check inputs
		if (alignment == null) {
			throw new IllegalArgumentException("Parameter 'rotationAngle' cannot be null.");		
		}

		// Assign Value
		m_Alignment = alignment;
		
		return;
	}
}
