package datamaxoneil.printer;

/**
 * This class represents the general parameters that can be used to alter the
 * printing characteristics when 'printing' to a DocumentDPL object. Printing
 * characteristics are 'local' to each print command they are passed into. To
 * have global settings the same parameters object can be passed into each write
 * call to the Document object.
 * 
 * @author Datamax-O'Neil
 * @version 2.1.0 (25 Oct 2013)
 */
public class ParametersDPL extends Parameters
{
		/** 
		 Value for rotating the printing by.  Rotations are counter-clockwise.
		*/
		public enum Rotation
		{

			/** 
			 Unrotated
			*/
			Rotate_0(1),

			/** 
			 Rotated 90 degrees counter-clockwise
			*/
			Rotate_90(2),

			/** 
			 Rotated 180 degrees counter-clockwise
			*/
			Rotate_180(3),

			/** 
			 Rotated 270 degrees counter-clockwise
			*/
			Rotate_270(4);

			private  int angle;

			private Rotation(int intValue)
			{
				angle = intValue;
			}

			public int getValue()
			{
				return angle;
			}

		}

		/** 
		 Alignment to use when drawing text
		*/
		public enum Alignment
		{

			/** 
			 Text will be aligned on the left starting at the x position specified.
			*/
			Left,

			/** 
			 Text will be center aligned at the x position specified.
			*/
			Center,

			/** 
			 Text will be aligned on the right starting at the x position specified.
			*/
			Right;

		}
		/** 
		 Measurement unit to be interpreted
		*/
		public enum MeasurementMode
		{
			/** 
			 Interpret values entered in meters
			*/
			Metric,
			/** 
			 Interpret values entered in inches
			*/
			Inch;

		}

		/** 
		 Double byte Symbol Sets for printing
		*/
		public enum DoubleByteSymbolSet
		{
			/** 
			 BIG 5(Taiwan) Encoded
			*/
			BIG5,

			/** 
			 EUC(Extended UNIX Code)
			*/
			EUC,
			/** 
			 Government Bureau Industry Standard,Chinese(PRC), default
			*/
			GovernmentBureau,
			/** 
			 Japanese Industry Standard, default
			*/
			JIS,
			/** 
			 Shift Japanese Industry Standard
			*/
			ShiftJIS,
			/** 
			 Unicode (including Korean)
			*/
			Unicode;

		}
		/** 
		 Single Byte Symbol Sets for printing
		*/
		public enum SingleByteSymbolSet
		{
			/** 
			 Desktop
			*/
			DeskTop,
			/** 
			 ISO 8859/1 Latin 1
			*/
			ISO8859_1_Latin1,
			/** 
			 ISO 8859/2 Latin 2
			*/
			ISO8859_2_Latin2,
			/** 
			 ISO 8859/9 Latin 5
			*/
			ISO8859_9_Latin5,
			/** 
			 ISO 8859/10 Latin 6
			*/
			ISO8859_10_Latin6,

			/** 
			 ISO 8859/15 Latin 9
			*/
			ISO8859_15_Latin9,

			/** 
			 ISO 15 Italian
			*/
			ISO15_Italian,

			/** 
			 Legal
			*/
			Legal,
			/** 
			 Math-8
			*/
			Math_8,

			/** 
			 Macintosh
			*/
			Macintosh,

			/** 
			 PS Math
			*/
			PS_Math,

			/** 
			 PC-858 MultiLingual
			*/
			PC858_Multi,
			/** 
			 PC-8,Code Page 437
			*/

			PC8_Code437,
			/** 
			 PC-8 D/N,Code Page 437N
			*/
			PC8_DN_Code437N,

			/** 
			 PC-852 Latin 2
			*/
			PC852_Latin2,

			/** 
			 PC-862 Latin/Hebrew
			*/
			PC862_LatinHebrew,

			/** 
			 PC-850 Multilingual
			*/
			PC850_Multi,

			/** 
			 PC-864 Latin/Arabic
			*/
			PC864_LatinArabic,

			/** 
			 PC-8 TK,Code Page 437T
			*/
			PC8TK_Code437T,
			/** 
			 PC-1004
			*/
			PC1004,
			/** 
			 PC-775 Baltic
			*/
			PC775_Baltic,
			/** 
			 Roman-8
			*/
			Roman8,
			/** 
			 Roman-9
			*/
			Roman9,
			/** 
			 ISO 17: Spanish
			*/
			ISO17_Spanish,
			/** 
			 ISO 11: Swedish
			*/
			ISO11_Swedish,
			/** 
			 PS Text
			*/
			PS_Text,
			/** 
			 ISO 4: United Kingdom
			*/
			ISO4_UK,
			/** 
			 ISO 6: ASCII
			*/
			ISO6_ASCII,
			/** 
			 UTF8
			*/
			UTF8,
			/** 
			 Ventura International
			*/
			VentInt,
			/** 
			 Ventura Math
			*/
			VentMath,
			/** 
			 Windows 3.1 Latin 1
			*/
			Windows31_Latin1,
			/** 
			 Windows Latin/Arabic
			*/
			Windows_LatinArabic,

			/** 
			 Windows 3.1 Latin 2
			*/
			Windows31_Latin2,

			/** 
			 Windows 3.1 Baltic
			*/
			Windows31_Baltic,

			/** 
			 Windows 3.0 Latin 1
			*/
			Windows30_Latin1,

			/** 
			 Windows Latin/Cyrillic
			*/
			Windows_LatinCyrillic,
			/** 
			 Windows 3.1 Latin 5
			*/
			Windows31_Latin5;

		}

		/** 
		 Single Byte Symbol Set to be used for printing
		*/
		private SingleByteSymbolSet m_SBSymbolSet = SingleByteSymbolSet.PC850_Multi;

		/** 
		 Double byte symbol set to be used for printing
		*/
		private DoubleByteSymbolSet m_DBSymbolSet = DoubleByteSymbolSet.GovernmentBureau;

		/** 
		 Specifies whether print job is using dot or point units. Usually used for specifying height/width
		 of scalable fonts.
		*/
		private boolean m_isDotMode = false;

		/** 
		 Specifies the ID of the object type we are printing(eg. font type, barcode, graphics)
		*/
		private String m_typeID = "0";

		/** 
		 Height of font to be print
		*/
		private int m_FontHeight = 0;


		/** 
		 Width of font to be print
		*/
		private int m_FontWidth = 0;

		/** 
		 Specifies whether to use single byte symbol set or double byte symbol set
		*/
		private boolean m_isUnicode = false;

		/** 
		 Degrees to rotate the printing.
		*/
		private Rotation m_Rotation = Rotation.Rotate_0;

		/** 
		 Alignment to use when drawing text
		*/
		private Alignment m_Alignment = Alignment.Left;

		/** 
		 Specifies whether the print job should be mirror
		*/
		private boolean m_IsMirrored = false;

		/** 
		 MEasurement units to be interpreted by printer
		*/
		private MeasurementMode m_Measurement = MeasurementMode.Inch;

		/** 
		 Height of symbol for barcode. 
		*/
		private int m_symbolHeight = 0;

		/** 
		 Fill pattern for graphics
		*/
		private int m_fillPattern = 0;

		/** 
		 Wide Bar width of barcode
		*/
		private int m_wideBarWidth = 0;

		/** 
		 Narrow Bar width of barcode
		*/
		private int m_narrowBarWidth = 0;

		/**
		 * This is the number of degrees the printing will be rotated about the
		 * starting location in the counter-clockwise direction. The default value
		 * for this parameter is 0 degrees.
		 * 
		 * @return The number of degrees the printing will be rotated
		 *         counter-clockwise.
		 * @since 1.1.0
		 */
		public Rotation getRotate() {
			return m_Rotation;
		}

		/**
		 * This is the horizontal alignment to use when writing the item. The
		 * options are:
		 * <ul>
		 * <li>Alignment.Left - Left aligned. This will write the item starting at
		 * the column specified in the write command.</li>
		 * <li>Alignment.Center - Center aligned. This will write the item centered
		 * at the column specified in the write command.</li>
		 * <li>Alignment.Right - Right aligned. This will write the item flush
		 * against the right side of the column specified in the write command.</li>
		 * </ul>
		 * 
		 * @return Alignment setting to use.
		 */
		public Alignment getAlignment() {
			return m_Alignment;
		}
		/** 
		 Wide Bar width of barcode
		*/
		public  int getWideBarWidth()
		{
			return m_wideBarWidth;
		}
		/**
		 * Set the WideBar width of barcode
		 * @param value - WideBar width value
		 */
		public  void setWideBarWidth(int value)
		{
			if ((value < 0) || (value > 61))
			{
					// Out of range
				throw new IllegalArgumentException(String.format("Parameter 'wideBarWidth' must be in the range of %1$s to %2$s, a value of %3$s was given.", 0, 61, value));
			}
			m_wideBarWidth = value;
		}

		/** 
		 Narrow Bar width of barcode
		*/
		public int getNarrowBarWidth()
		{
			return m_narrowBarWidth;
		}
		/**
		 * Set the NarrowBar width of barcode
		 * @param value - NarrowBar width value
		 */
		public void setNarrowBarWidth(int value)
		{
				// Check inputs
			if ((value < 0) || (value > 61))
			{
					// Out of range
				throw new IllegalArgumentException(String.format("Parameter 'narrowBarWidth' must be in the range of %1$s to %2$s, a value of %3$s was given.", 0, 61, value));
			}
			m_narrowBarWidth = value;
		}


		/** 
		 Specifies whether to use single byte symbol set or double byte symbol set
		*/
		public boolean getIsUnicode()
		{
			return m_isUnicode;
		}
		/**Set whether to use single byte symbol set or double byte symbol set
		 * @param value - true = use double byte symbol set, false = use single byte symbol set
		 */
		public void setIsUnicode(boolean value)
		{
			m_isUnicode = value;
		}

		/** 
		 Specifies the ObjectType ID we are printing(eg. font type, barcode, graphics)
		*/
		public String getTypeID()
		{
			return m_typeID;
		}
		/**
		 * Set One character object type ID
		 * @param value - ID of object. Must be 1 character in length
		 */
		public void setTypeID(String value)
		{
			if (value.length() != 1)
			{
				throw new IllegalArgumentException(String.format("Parameter 'typeID' must be between 1 character in length. Value given was %1$s characters long.", value.length()));
			}
				//Assign value
			m_typeID = value;
		}


		/** 
		 Single Byte Symbol Set to be used for printing
		*/
		public SingleByteSymbolSet getSBSymbolSet()
		{
			return m_SBSymbolSet;
		}
		/** 
		 Set Single Byte Symbol Set to be used for printing
		 @param value - symbol set value
		*/
		public void setSBSymbolSet(SingleByteSymbolSet value)
		{
			m_SBSymbolSet = value;
		}


		/** 
		 Double byte symbol set to be used for printing
		*/
		public DoubleByteSymbolSet getDBSymbolSet()
		{
			return m_DBSymbolSet;
		}
		/** 
		 Set Single Byte Symbol Set to be used for printing
		 @param value - symbol set value
		*/
		public  void setDBSymbolSet(DoubleByteSymbolSet value)
		{
			m_DBSymbolSet = value;
		}

		/** 
		 Converts the specified symbol set to its corresponding string value
		 
		 @param symbolSet - Single Byte Symbol set to convert
		 @return String value of the symbol Set
		*/
		public  String SymbolSetToString(SingleByteSymbolSet symbolSet)
		{
			switch (symbolSet)
			{
				case DeskTop:
					return "DT";
				case ISO8859_1_Latin1:
					return "ET";
				case ISO8859_2_Latin2:
					return "E2";
				case ISO8859_9_Latin5:
					return "E5";
				case ISO8859_10_Latin6:
					return "E6";
				case ISO8859_15_Latin9:
					return "E9";
				case ISO15_Italian:
					return "IT";
				case Legal:
					return "LG";
				case Math_8:
					return "M8";
				case Macintosh:
					return "MC";
				case PC858_Multi:
					return "P9";
				case PC8_Code437:
					return "PC";
				case PC8_DN_Code437N:
					return "PD";
				case PC852_Latin2:
					return "PE";
				case PC862_LatinHebrew:
					return "PH";
				case PC850_Multi:
					return "PM";
				case PC864_LatinArabic:
					return "PR";
				case PC8TK_Code437T:
					return "PT";
				case PC1004:
					return "PU";
				case PC775_Baltic:
					return "PV";
				case Roman8:
					return "R8";
				case Roman9:
					return "R9";
				case ISO17_Spanish:
					return "SP";
				case ISO11_Swedish:
					return "SW";
				case PS_Text:
					return "TS";
				case ISO4_UK:
					return "UK";
				case ISO6_ASCII:
					return "US";
				case UTF8:
					return "U8";
				case VentInt:
					return "VI";
				case VentMath:
					return "VM";
				case Windows31_Latin1:
					return "W1";
				case Windows_LatinArabic:
					return "WA";
				case Windows31_Latin2:
					return "WE";
				case Windows31_Baltic:
					return "WL";
				case Windows30_Latin1:
					return "WO";
				case Windows_LatinCyrillic:
					return "WR";
				case Windows31_Latin5:
					return "WT";
				default:
					return "US";

			}
		}

		/** 
		 Converts the specified symbol set to its corresponding string value
		 
		 @param symbolSet Double Byte Symbol set to convert
		 @return String value of the symbol Set
		*/
		public String SymbolSetToString(DoubleByteSymbolSet symbolSet)
		{
			switch (symbolSet)
			{
				case EUC:
					return "EU";
				case GovernmentBureau:
					return "GB";
				case JIS:
					return "JS";
				case ShiftJIS:
					return "SJ";
				case Unicode:
					return "UC";
				default:
					return "GB";

			}
		}

		/** 
		 * Height of font to be print
		 */
		public int getFontHeight()
		{
			return m_FontHeight;
		}
		/**
		 * Set fill pattern for graphics
		 * @param value - Font height value.
		 */
		public void setFontHeight(int value)
		{
				//Validate parameters
			if (value != 0 && (value < 4 || value > 4163))
			{
				throw new IllegalArgumentException(String.format("Parameter 'fontHeight' must be equal to 0 or between 4 - 4163. Value given was %1$s", value));
			}
			m_FontHeight = value;
		}
		/** 
		 * Width of font to be print
		 */
		public int getFontWidth()
		{
			return m_FontWidth;
		}
		/** 
		 * Set width of font to be print
		 * @param value - Font Width value.
		 */
		public void setFontWidth(int value)
		{
			if (value != 0 && (value < 4 || value > 4163))
			{
				throw new IllegalArgumentException(String.format("Parameter 'fontWidth' must be  0 or between 4 - 4163. Value given was %1$s", value));
			}
			m_FontWidth = value;
		}

		/** 
		 * Fill pattern for graphics
		 * */
		public int getFillPattern()
		{
			return m_fillPattern;
		}
		/**
		 * Set fill pattern for graphics
		 * @param value - Valid fill pattern values are 0-11 
		 */
		public void setFillPattern(int value)
		{
			if (value < 0 || value > 11)
			{
				throw new IllegalArgumentException(String.format("Parameter 'fillPattern' must be between 0 - 11. Value given was %1$s", value));
			}
			m_fillPattern = value;
		}

		/** 
		 Specifies whether print job is using dot or point units. Usually used for specifying height/width
		 of scalable fonts.
		*/
		public boolean getIsDotMode()
		{
			return m_isDotMode;
		}
		/**
		 * Set the dot mode for print job
		 * @param value - true = dot units, false = points unit
		 */
		public void setIsDotMode(boolean value)
		{
			m_isDotMode = value;
		}

		/** 
		 * Height of symbol for barcode. 
		 */
		public int getSymbolHeight()
		{
			return m_symbolHeight;
		}
		/**
		 *  Set Height of symbol for barcode.
		 *  @param value - Height of symbol. Valid values 0-999.
		 */
		public void setSymbolHeight(int value)
		{
				//Validate symbol height
			if (value < 0 || value > 999)
			{
				throw new IllegalArgumentException(String.format("Parameter 'symbolHeight' must be between 0 - 999. Value given was %1$s", value));
			}

			m_symbolHeight = value;
		}




		/** 
		 * The horizontal multiplier property will cause the horizontal printing
		 * of the object to be scaled by that amount.  So a value of two would
		 * cause a printing of the object to be double its original width.  The
		 * default value for this parameter is 1 and the range of valid values
		 * is 1 to 61.
		 * @param value Factor by which to scale the printing horizontally.
		 * @throws IllegalArgumentException If the parameter values are not in the
		 *         valid range.
		 * @since 1.0.0
		 */ 
		public void setHorizontalMultiplier(int value)
		{
			setHorizontalMultiplier(value, 0, 61);
		}


		/**
		 * The vertical multiplier property will cause the vertical printing of the
		 * object to be scaled by that amount. So a value of two would cause a
		 * printing of the object to be double its original hieght. The default
		 * value for this parameter is 1 and the range of valid values is 1 to 255.
		 * 
		 * @param value Factor by which to scale the printing vertically.
		 * @throws IllegalArgumentException If the parameter values are not in the
		 *         valid range.
		 * @since 1.0.0
		 */

		public void setVerticalMultiplier(int value)
		{
			setVerticalMultiplier(value, 0, 61);
		}

		/**
		 * This is the number of degrees the printing will be rotated about the
		 * starting location in the counter-clockwise direction. The default value
		 * for this parameter is 0 degrees.
		 * 
		 * @param rotationAngle Number of degrees to rotate the printing the range
		 *        of valid values is 0, 90, 180 or 270 degrees.
		 * @throws IllegalArgumentException If the parameter values are not in the
		 *         valid range.
		 * @since 2.1.0
		 */

		public void setRotate(Rotation rotationAngle) throws IllegalArgumentException {

			// Check inputs
			if (rotationAngle == null) {
				throw new IllegalArgumentException("Parameter 'rotationAngle' cannot be null.");		
			}

			// Assign Value
			m_Rotation = rotationAngle;

			return;
		}

		/**
		 * This is the horizontal alignment to use when writing the item. The
		 * options are:
		 * <ul>
		 * <li>Alignment.Left - Left aligned. This will write the item starting at
		 * the column specified in the write command.</li>
		 * <li>Alignment.Center - Center aligned. This will write the item centered
		 * at the column specified in the write command.</li>
		 * <li>Alignment.Right - Right aligned. This will write the item flush
		 * against the right side of the column specified in the write command.</li>
		 * </ul>
		 * 
		 * @param alignment Alignment setting to use.
		 */
		public void setAlignment(Alignment alignment) {

			// Check inputs
			if (alignment == null) {
				throw new IllegalArgumentException("Parameter 'rotationAngle' cannot be null.");		
			}

			// Assign Value
			m_Alignment = alignment;
			
			return;
		}

		/** 
		 Get the current value of isMirrored
		*/
		public boolean getIsMirrored()
		{
			return m_IsMirrored;
		}
		/**
		 * Specifies whether print job should be mirrored or not
		 * @param value - Set to true for mirrored and false for no mirrored. 
		 */
		public void setIsMirrored(boolean value)
		{
			m_IsMirrored = value;
		}

		/** 
		 Measurement unit to be interpreted by printer.
		*/
		public MeasurementMode getMeasurement()
		{
			return m_Measurement;
		}
		/** 
		 Set measurement unit to be interpreted by printer.
		*/
		public void setMeasurement(MeasurementMode value)
		{
			m_Measurement = value;
		}

}