package datamaxoneil.printer;

/**
 * This base class represents the general parameters that can be used to alter
 * the printing characteristics when 'printing' to a Document object. Printing
 * characteristics are 'local' to each print command they are passed into. To
 * have global settings the same parameters object can be passed into each write
 * call to the Document object.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public abstract class Parameters {

	/** Name of the font to print with. */
	private String m_FontName = "";

	/** Should the entire image be printed in 'reverse video'. */
	private boolean m_IsInverse = false;

	/** Integer value to scale all printing horizontally. */
	private int m_HorizontalMultiplier = 1;

	/** Integer value to scale all printing vertically. */
	private int m_VerticalMultiplier = 1;

	/**
	 * The font name is a character string that represents the name of the font,
	 * already loaded on the printer, that you wish to print with.
	 * 
	 * @return The font name or the empty string if the default printer font is
	 *         to be used.
	 * @since 1.0.0
	 */
	public String getFont() {
		return m_FontName;
	}

	/**
	 * The font name is a character string that represents the name of the font,
	 * already loaded on the printer, that you wish to print with.
	 * 
	 * @param fontName The font name or the empty string if the default printer
	 *        font is to be used.
	 * @param size The size the font name must be. The empty string is also
	 *        accepted and that will 'clear' the value.
	 * @throws IllegalArgumentException If the parameter values are not in the
	 *         valid range.
	 * @since 1.0.0
	 */
	protected void setFont(String fontName, int size) throws IllegalArgumentException {

		// Check inputs
		if ((fontName == null) || ((fontName.length() != size) && (fontName.length() != 0))) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'fontName' must be " + "either " + size
					+ " characters in length or the empty string.");
		}

		// Assign Value
		m_FontName = fontName;

		return;
	}
	/**
	 * The font name is a character string that represents the name of the font,
	 * already loaded on the printer, that you wish to print with.
	 * @param fontName the font name or the empty string if the default printer font is to be used
	 * @throws IllegalArgumentException If the parameter values are not in the
	 *         valid range.
	 * @since 2.1.0 
	 */
	protected void setFont(String fontName)
	{
			// Check inputs
		if ((fontName == null) || ((fontName.length() > 3) && (fontName.length() != 0)))
		{
				// Out of range
			throw new IllegalArgumentException(String.format("Parameter 'fontName' must be at least " + Integer.toString(3)
				+ " characters in length or the empty string."));
		}
		// Assign Value
		m_FontName = fontName;
	}

	/**
	 * The inverse property, if set to true, will cause the image to print white
	 * on a black background instead of the traditional black on a white
	 * background. The default value for this property is false.
	 * 
	 * @return The value 'true' if the property is set or 'false' otherwise.
	 * @since 1.1.0
	 */
	public boolean getIsInverse() {
		return m_IsInverse;
	}

	/**
	 * The inverse property, if set to true, will cause the image to print white
	 * on a black background instead of the traditional black on a white
	 * background. The default value for this property is false.
	 * 
	 * @param isPrintingInverse A boolean which is set to true if you wish to
	 *        print inverse or false if you wish to print normally.
	 * @since 1.1.0
	 */
	public void setIsInverse(boolean isPrintingInverse) {
		m_IsInverse = isPrintingInverse;
		return;
	}

	/**
	 * The horizontal multiplier property will cause the horizontal printing of
	 * the object to be scaled by that amount. So a value of two would cause a
	 * printing of the object to be double its original width. The default value
	 * for this parameter is 1.
	 * 
	 * @return The value the horizontal dimension will be scaled by.
	 * @since 1.0.0
	 */
	public int getHorizontalMultiplier() {
		return m_HorizontalMultiplier;
	}

	/**
	 * The horizontal multiplier property will cause the horizontal printing of
	 * the object to be scaled by that amount. So a value of two would cause a
	 * printing of the object to be double its original width. The default value
	 * for this parameter is 1.
	 * 
	 * @param multiplier The value the horizontal dimension should be scaled by.
	 * @param lowerBound The lower allowed bound.
	 * @param upperBound The upper allowed bound.
	 * @throws IllegalArgumentException If the parameter values are not in the
	 *         valid range.
	 * @since 1.0.0
	 */
	protected void setHorizontalMultiplier(int multiplier, int lowerBound, int upperBound) throws IllegalArgumentException {

		// Check inputs
		if ((multiplier < lowerBound) || (multiplier > upperBound)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'multiplier' must be in "
					+ "the range of " + lowerBound + " to " + upperBound + ", a value of "
					+ multiplier + " was given.");
		}

		// Assign Value
		m_HorizontalMultiplier = multiplier;

		return;
	}

	/**
	 * The vertical multiplier property will cause the vertical printing of the
	 * object to be scaled by that amount. So a value of two would cause a
	 * printing of the object to be double its original height. The default
	 * value for this parameter is 1.
	 * 
	 * @return The value the vertical dimension will be scaled by.
	 * @since 1.0.0
	 */
	public int getVerticalMultiplier() {
		return m_VerticalMultiplier;
	}

	/**
	 * The vertical multiplier property will cause the vertical printing of the
	 * object to be scaled by that amount. So a value of two would cause a
	 * printing of the object to be double its original height. The default
	 * value for this parameter is 1. This is the internal version of the
	 * setting function to allow derived classes to pass in the value and the
	 * range for centralized checking.
	 * 
	 * @param multiplier The value the vertical dimension will be scaled by.
	 * @param lowerBound The lower allowed bound.
	 * @param upperBound The upper allowed bound.
	 * @throws IllegalArgumentException If the parameter values are not in the
	 *         valid range.
	 * @since 1.0.0
	 */
	protected void setVerticalMultiplier(int multiplier, int lowerBound, int upperBound) throws IllegalArgumentException {

		// Check inputs
		if ((multiplier < lowerBound) || (multiplier > upperBound)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'multiplier' must be in "
					+ "the range of " + lowerBound + " to " + upperBound + ", a value of "
					+ multiplier + " was given.");
		}

		// Assign Value
		m_VerticalMultiplier = multiplier;

		return;
	}
}
