package datamaxoneil.printer;

import java.io.ByteArrayOutputStream;
import java.awt.image.BufferedImage;

/**
 * This class handels the interface between the printer and the user application
 * that wishes to use an Datamax-O'Neil printer. These functions can be used to 'build'
 * a document using absctract methods instead of the raw 'LP Print' mode
 * commands. A simple example is:<BR>
 * <BR>
 * <CODE><PRE>
 *    // Print Hello World
 *    DocumentLP docLP;
 *    docLP = new DocumentLP("$");
 *    docLP.writeText("Hello World");
 * </PRE></CODE> This document can then be sent to a PrinterONeil object through
 * the print method to print out.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class DocumentLP extends Document {

	/** Different character sets you can use to remap a font */
	public enum CharacterSet {
		/** This uses the USA characters in a CP437 224 character font. */
		USA('0', "USA"),
		/** This uses the French characters in a CP437 224 character font. */
		France('1', "France"),
		/** This uses the German characters in a CP437 224 character font. */
		Germany('2', "Germany"),
		/** This uses the British characters in a CP437 224 character font. */
		UK('3', "UK"),
		/** This uses the Danish characters in a CP437 224 character font. */
		Denmark('4', "Denmark"),
		/** This uses the Swedish characters in a CP437 224 character font. */
		Sweden('5', "Sweden"),
		/** This uses the Italian characters in a CP437 224 character font. */
		Italy('6', "Italy"),
		/** This uses the Spanish characters in a CP437 224 character font. */
		Spain('7', "Spain");
				
		/** External Value */
		private final char m_Value;
		
		/** User Friendly Name */
		private final String m_Name;
		
		/**
		 * Constructor
		 * @param value External Value
		 * @param name User Friendly Name
		 */
		CharacterSet(char value, String name) { m_Value = value; m_Name = name; }	
				
		/**
		 * External Value
		 * @return External Value
		 */
		public char value() { return m_Value; }
				
		/**
		 * User Friendly Name
		 * @return User Friendly Name
		 */
		public String named() { return m_Name; }	
	};

	/** The number of additional dotlines between printed rows. */
	private int m_LineSpacing = 0;

	/** The length of a "page" in text lines */
	private int m_FormLength = 0;

	/** CP437 224 character font remapping value */
	private CharacterSet m_CharacterRemap = CharacterSet.USA;

	/**
	 * This constructor will create a blank Document object and set the default
	 * font to use to the one specified.
	 * 
	 * @param fontName The one character font name.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public DocumentLP(String fontName) throws IllegalArgumentException {

		// Set Parameters
		m_FontNameLength = 1;

		// Assign Value
		setDefaultFont(fontName);

		return;
	}

	/**
	 * This is the number of text lines per 'page'. This value is used when the
	 * advanceToNextPage() function is called. The total height is based on the
	 * currently selected font.
	 * 
	 * @return The number of text lines per page. The default page size is 0
	 *         which means to use the printer setting.
	 * @since 1.0.0
	 */
	public int getPageLength() {
		return m_FormLength;
	}

	/**
	 * This is the number of text lines per 'page'. This value is used when the
	 * advanceToNextPage() function is called. The total height is based on the
	 * currently selected font.
	 * 
	 * @param textLines The number of text lines per page. The default page size
	 *        is 0 which means to use the printer setting. Valid values are from
	 *        0 to 255.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void setPageLength(int textLines) throws IllegalArgumentException {

		// Check inputs
		if ((textLines < 0) || (textLines > 255)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'textLines' must be "
					+ "from 0 to 255, a value of " + textLines + " was given.");
		}

		// Assign Value
		m_FormLength = textLines;

		return;
	}

	/**
	 * This is the number of dotlines to add in between each text line.
	 * 
	 * @return The number of 1/8mm dotlines to put between each text line. The
	 *         default value is 0.
	 * @since 1.0.0
	 */
	public int getLineSpacing() {
		return m_LineSpacing;
	}

	/**
	 * This is the number of dotlines to add in between each text line.
	 * 
	 * @param dotLines The number of 1/8mm dotlines to put between each text
	 *        line. The default value is 0 and the range of values is 0 to 255.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void setLineSpacing(int dotLines) throws IllegalArgumentException {

		// Check inputs
		if ((dotLines < 0) || (dotLines > 255)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'dotLines' must be "
					+ "from 0 to 255, a value of " + dotLines + " was given.");
		}

		// Assign Value
		m_LineSpacing = dotLines;

		// Add to document
		byte commandValue = (byte) dotLines;
		addToDoc(m_Document, ESC + "A");
		addToDoc(m_Document, commandValue);

		return;
	}

	/**
	 * This will cause some of the international characters, when using a CP437
	 * 224 character font, to be remapped. The default value for this is
	 * CharacterSet.USA.
	 * 
	 * @return The mapping of which CP437 224 character fonts to use.
	 * @since 1.0.0
	 */
	public CharacterSet getCharacterRemap() {
		return m_CharacterRemap;
	}

	/**
	 * This will cause some of the international characters, when using a CP437
	 * 224 character font, to be remapped. The default value for this is
	 * CharacterSet.USA. 
	 * 
	 * @param characterSet The mapping of which CP437 224 character fonts to
	 *        use.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void setCharacterRemap(CharacterSet characterSet) throws IllegalArgumentException {

		// Check inputs
		if (characterSet == null) {
			throw new IllegalArgumentException("Parameter 'characterSet' cannot be null.");
		}

		// Assign Value
		m_CharacterRemap = characterSet;

		return;
	}

	/**
	 * This function will print the provided text, appending a line feed/carrage
	 * return, to the document object using the default printing parameter
	 * values.
	 * 
	 * @param textString This is the text you wish to print.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeText(String textString) throws IllegalArgumentException {

		// Call full function
		writeText(textString, new ParametersLP());

		return;
	}

	/**
	 * This function will print the provided text, appending a line feed/carriage
	 * return, to the document object using the provided printing parameter
	 * values.
	 * 
	 * @param textString This is the text you wish to print.
	 * @param parameters This ParametersLP object specifies any printing
	 *        parameter values you wish to alter for the printing of this item.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeText(String textString, ParametersLP parameters) throws IllegalArgumentException {

		// Check inputs
		if (textString == null) {
			// Invalid text object
			throw new IllegalArgumentException("Parameter 'textString' was null.");
		}

		if (parameters == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}

		// Add to Document
		byte[] paramString = parameterString(parameters, PrintingType.General);
		byte[] paramEndString = parameterEndString(parameters, PrintingType.General);

		addToDoc(m_Document, paramString);
		addToDoc(m_Document, textString);
		addToDoc(m_Document, paramEndString);
		addToDoc(m_Document, EOL);

		return;
	}

	/**
	 * This function will print the provided text to the document object using
	 * the default printing parameter values.
	 * 
	 * @param textString This is the text you wish to print.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeTextPartial(String textString) throws IllegalArgumentException {
		writeTextPartial(textString, new ParametersLP());
		return;
	}

	/**
	 * This function will print the provided text to the document object using
	 * the provided printing parameter values.
	 * 
	 * @param textString This is the text you wish to print.
	 * @param parameters This ParametersLP object specifies any printing
	 *        parameter values you wish to alter for the printing of this item.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeTextPartial(String textString, ParametersLP parameters) throws IllegalArgumentException {

		// Check inputs
		if (textString == null) {
			// Invalid text object
			throw new IllegalArgumentException("Parameter 'textString' was null.");
		}

		if (parameters == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}

		// Add to Document
		byte[] paramString = parameterString(parameters, PrintingType.General);
		byte[] paramEndString = parameterEndString(parameters, PrintingType.General);

		addToDoc(m_Document, paramString);
		addToDoc(m_Document, textString);
		addToDoc(m_Document, paramEndString);

		return;
	}

	/**
	 * This will cause the image specified to be printed. Images will be
	 * expanded to occupy the entire width of the printer, so the correct
	 * current width of the printer must be specified. Images that are too wide
	 * will be cropped, and images that are too narrow will be padded on the
	 * right.
	 * 
	 * @param imageObject Image to print
	 * @param printHeadWidth Width of the print head in dots. Usually 384 for 2
	 *        inch printers, 576 for 3 inch and 832 for 4 inch.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.1.0
	 */
	public void writeImage(BufferedImage imageObject, int printHeadWidth) throws IllegalArgumentException {

		// Check inputs
		if (imageObject == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'imageObject' was null.");
		}

		if (printHeadWidth < 1) {
			// Invalid printHeadWidth object
			throw new IllegalArgumentException("Parameter 'printHeadWidth' must be greater than 0.");
		}

		// Convert to 1 Byte per pixel image
		int height = imageObject.getHeight();
		int width = imageObject.getWidth();

		// Array Sizes
		byte blanklineCount = 0;
		byte[] dataline = new byte[(printHeadWidth + 7) >> 3];
		int[] imageData = new int[height * width];
		/*Android version*/
		//imageObject.getPixels(imageData, 0, width, 0, 0, width, height);
		/*Java version*/
		imageObject.getRGB(0, 0, width, height, imageData, 0, width);
		// Start the RLE compressed graphic image
		addToDoc(m_Document, ESC + "B");
		
		// Loop through each row
		for (int row = 0; row < height; row++) {
			boolean blankLine = true;

			// Get the current scanline
			for (int index = 0; index < width; index += 8) {
				int value;
				boolean set;
				byte currentByte = 0;
				int offset = (row * width) + index;

				// Check for EOL
				if (index >= printHeadWidth)
					break;

				value = ((index + 0) < width) ? imageData[offset + 0] & 0x00FFFFFF : 0x00FFFFFF;
				set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
				currentByte |= (set) ? (byte) 0x80 : (byte) 0x00;

				value = ((index + 1) < width) ? imageData[offset + 1] & 0x00FFFFFF : 0x00FFFFFF;
				set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
				currentByte |= (set) ? (byte) 0x40 : (byte) 0x00;

				value = ((index + 2) < width) ? imageData[offset + 2] & 0x00FFFFFF : 0x00FFFFFF;
				set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
				currentByte |= (set) ? (byte) 0x20 : (byte) 0x00;

				value = ((index + 3) < width) ? imageData[offset + 3] & 0x00FFFFFF : 0x00FFFFFF;
				set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
				currentByte |= (set) ? (byte) 0x10 : (byte) 0x00;

				value = ((index + 4) < width) ? imageData[offset + 4] & 0x00FFFFFF : 0x00FFFFFF;
				set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
				currentByte |= (set) ? (byte) 0x08 : (byte) 0x00;

				value = ((index + 5) < width) ? imageData[offset + 5] & 0x00FFFFFF : 0x00FFFFFF;
				set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
				currentByte |= (set) ? (byte) 0x04 : (byte) 0x00;

				value = ((index + 6) < width) ? imageData[offset + 6] & 0x00FFFFFF : 0x00FFFFFF;
				set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
				currentByte |= (set) ? (byte) 0x02 : (byte) 0x00;

				value = ((index + 7) < width) ? imageData[offset + 7] & 0x00FFFFFF : 0x00FFFFFF;
				set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
				currentByte |= (set) ? (byte) 0x01 : (byte) 0x00;

				// Now set in dataline
				dataline[index >> 3] = currentByte;
				blankLine &= (currentByte == 0x00);
			}

			// Check for blank line
			if (blankLine == false) {
				// We found some data so write out the blanks
				if (blanklineCount > 0) {
					addToDoc(m_Document, "A");
					addToDoc(m_Document, blanklineCount);
					blanklineCount = 0;
				}

				// Compress it and add to document
				addToDoc(m_Document, compressGraphicLine(dataline));
			}
			else {
				// If we made it this far its blank
				blanklineCount++;

				// Write out blanks if size is too big
				if (blanklineCount == 255) {
					addToDoc(m_Document, "A");
					addToDoc(m_Document, blanklineCount);
					blanklineCount = 0;
				}
			}
		}

		// Write out any trailing blanks
		if (blanklineCount > 0) {
			addToDoc(m_Document, "A");
			addToDoc(m_Document, blanklineCount);
			blanklineCount = 0;
		}

		// Finish the RLE compressed graphic image
		addToDoc(m_Document, ESC + "E");

		return;
	}

	/**
	 * This will compress the bitmap data with our RLE scheme. The optimal
	 * encoding will be used.
	 * 
	 * @param dataline Uncompressed image line
	 * @return Array of bytes that is the image line in RLE format.
	 */
	private byte[] compressGraphicLine(byte[] dataline) {
		byte count = 0;
		byte currentByte = 0;
		ByteArrayOutputStream rleString = new ByteArrayOutputStream(128);

		// Initialize as compressed
		addToDoc(rleString, "G");

		// Create rleString
		for (int index = 0; index < dataline.length; index++) {

			// Calculate the runs
			if (count == 0) {
				// If first then set
				currentByte = dataline[index];
				addToDoc(rleString, currentByte);
				count++;
			}
			else if ((count < 255) && (currentByte == dataline[index])) {
				// We are in a run
				count++;
			}
			else {
				// Something new or our count is big
				addToDoc(rleString, count);
				count = 0;

				// Start new run
				currentByte = dataline[index];
				addToDoc(rleString, currentByte);
				count++;
			}
		}

		// Write any trailing elements
		if (count > 0) {
			addToDoc(rleString, count);
		}

		// Compare it to uncompressed string
		if (rleString.size() > (dataline.length + 1)) {
			// Uncompressed is smaller
			rleString.reset();
			addToDoc(rleString, "U");

			for (int item = 0; item < dataline.length; item++) {
				addToDoc(rleString, dataline[item]);
			}
		}

		return rleString.toByteArray();
	}

	/**
	 * This command will causes a form feed to occur from the papers current
	 * location. The size of a form can be set with the setPageLength()
	 * function.
	 * 
	 * @since 1.0.0
	 */
	public void advanceToNextPage() {
		byte command = 0x0C;

		// Add to document
		addToDoc(m_Document, command);

		return;
	}

	/**
	 * This will advance the paper to the next queue mark and an additional
	 * number of dotlines as specified by the input parameter.
	 * 
	 * @param additionalRows Number of dotlines to move beyond the queue mark.
	 *        Each dotline is 1/8mm high and valid values are 0 <= n <= 65535.
	 * @since 1.0.0
	 */
	public void advanceToQueueMark(int additionalRows) {
		// Validate input
		if ((additionalRows < 0) || (additionalRows > 65535)) {
			throw new IllegalArgumentException("Parameter 'additionalRows' "
					+ "must be an integer from 0 to 65535, a value of " + additionalRows
					+ " was given.");
		}

		// Add to document
		char bitH = (char) ((additionalRows >> 8) & 0xFF);
		char bitL = (char) ((additionalRows >> 0) & 0xFF);

		addToDoc(m_Document, ESC + "Q" + bitL + bitH);

		return;
	}

	/**
	 * This will build the parameters string for the current type of object
	 * being rendered.
	 * 
	 * @param parameters Parameter object to get settings from.
	 * @param type Type of object being rendered.
	 * @return The parameters encoded into the printer's language for the
	 *         current object.
	 */
	private byte[] parameterString(ParametersLP parameters, PrintingType type) {
		ByteArrayOutputStream paramData = new ByteArrayOutputStream(128);

		// Set Font
		if (type == PrintingType.General) {
			if (parameters.getFont() == "")
				addToDoc(paramData, ESC + "w" + getDefaultFont());
			else
				addToDoc(paramData, ESC + "w" + parameters.getFont());
		}

		// Inverse Printing
		if (parameters.getIsInverse()) {
			// Update Inverse Settings
			addToDoc(paramData, ESC + "I1");
		}

		// Multipliers Fields
		if ((parameters.getHorizontalMultiplier() == 2)
				|| (parameters.getVerticalMultiplier() == 2)) {
			// Update Doubling Settings
			char bitField = 0;

			bitField |= (parameters.getHorizontalMultiplier() == 2) ? 0x20 : 0x00;
			bitField |= (parameters.getVerticalMultiplier() == 2) ? 0x10 : 0x00;
			addToDoc(paramData, ESC + "!" + bitField);
		}

		return paramData.toByteArray();
	}

	/**
	 * This will build the end parameters string for the current type of object
	 * being rendered.
	 * 
	 * @param parameters Parameter object to get settings from.
	 * @param type Type of object being rendered.
	 * @return The parameters encoded into the printer's language for the
	 *         current object.
	 */
	private byte[] parameterEndString(ParametersLP parameters, PrintingType type) {
		ByteArrayOutputStream paramData = new ByteArrayOutputStream(128);

		// Inverse Printing
		if (parameters.getIsInverse()) {
			// Revert Inverse Settings
			addToDoc(paramData, ESC + "I0");
		}

		// Multipliers Fields
		if ((parameters.getHorizontalMultiplier() == 2)
				|| (parameters.getVerticalMultiplier() == 2)) {
			// Revert Doubling Settings
			addToDoc(paramData, ESC + "!\0");
		}

		return paramData.toByteArray();
	}

	/**
	 * This method will return the rendered document in the printer's language
	 * for sending directly to the printer.
	 * 
	 * @return Rendered document in the printer's language.</returns>
	 * @since 1.1.0
	 */
	@Override
	public byte[] getDocumentData() {
		String globalsStart;
		String globalsEnd;
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();

		// Form global parameter settings
		globalsStart = "";
		globalsEnd = "";

		if (getPageLength() != 0) {
			char commandValue = (char) getPageLength();

			// Use Default Font for line height
			globalsStart += ESC + "w" + getDefaultFont();
			
			// Set our line length
			globalsStart += ESC + "C" + commandValue;
		}

		if (getLineSpacing() != 0) {
			char commandValue = (char) getLineSpacing();

			globalsStart += ESC + "A" + commandValue;
		}

		if (getCharacterRemap() != CharacterSet.USA) {
			char commandValue = (char)(getCharacterRemap().value() - '0');

			globalsStart += ESC + "R" + commandValue;
		}

		if (getIsLandscapeMode()) {
			char commandValue = (char) 0xB3;

			globalsStart += ESC + "L";
			globalsEnd += ESC + "" + commandValue;
		}
		
		// Use the @ command to reset everything because some are different
		// across thermal and impact.  Plus this works for new parameters too.
		globalsEnd += ESC + "@";
		
		// Build
		addToDoc(dataStream, ESC + "EZ{}{LP}" + globalsStart);
		addToDoc(dataStream, m_Document.toByteArray());
		addToDoc(dataStream, globalsEnd);

		return dataStream.toByteArray();
	}
}
