package datamaxoneil.printer;

import java.io.ByteArrayOutputStream;
import datamaxoneil.printer.ParametersEZ.Alignment;
import datamaxoneil.printer.ParametersEZ.Rotation;

/**
 * This class handles the interface between the printer and the user application
 * that wishes to use an Datamax-O'Neil printer. These functions can be used to 'build'
 * a document using abstract methods instead of the raw 'EZ Print' mode
 * commands. A simple example is:<BR>
 * <BR>
 * <CODE><PRE>
 *    // Print Hello World
 *    DocumentEZ docEZ;
 *    docEZ = new DocumentEZ("$");
 *    docEZ.writeText("Hello World",1,1);
 * </PRE></CODE> This document can then be sent to a PrinterONeil object through
 * the print method to print out.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class DocumentEZ extends Document {

	/** QStop options for use when feeding the paper */
	public enum QStop {
		/**
		 * The printer will act according to the value programmed into its internal
		 * flash memory.
		 */
		Default,	
		/**
		 * The printer will stop printing the set number of dotlines after the queue
		 * mark is found on the top of the media. This mode applies to the LP3, MF2,
		 * MF3, 2t and 4t.
		 */
		Front, 	
		/**
		 * The printer will stop printing the set number of dotlines after the queue
		 * marks is found on the back of the media. This mode applies to the LP3.
		 */
		Back,	
		/**
		 * The printer will stop printing the set number of dotlines after the
		 * inter-label gap is found between labels. This mode applies to the LP3.
		 */
		Gap,	
		/**
		 * The printer will ignore all queue stop settings in its internal flash
		 * memory and just print normally.
		 */
		None;
	}
	
	/** The initial number of dotlines to back-up before printing. */
	private int m_InitialPaperBack = 0;

	/** The maximum number of dotlines to allow to be printed. */
	private int m_PageLength = 0;

	/** The number of copies to print. */
	private int m_PrintQuantity = 1;

	/** Should printing be 'optimized' for back-up dotlines */
	private boolean m_PrintOptimize = false;

	/** The queue mark behavior mode. */
	private QStop m_QMarkMode = QStop.Default;

	/** Number of dotlines past the queue mark to stop feeding paper. */
	private int m_QMarkDotLines = 0;

	/**
	 * This constructor takes the name for a font to use when none is specified
	 * for write operations.
	 * 
	 * @param fontName The five character font name.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public DocumentEZ(String fontName) throws IllegalArgumentException {

		// Set Parameters
		m_FontNameLength = 5;

		// Assign Value
		setDefaultFont(fontName);

		return;
	}

	/**
	 * This is the amount of dotlines, each 1/8mm, the paper will be 'backed-up'
	 * before printing is started. The default value is 0 dotlines.
	 * 
	 * @return The number of dotlines, each 1/8mm, to initially back-up.
	 * @since 1.0.0
	 */
	public int getInitialPaperBackup() {
		return m_InitialPaperBack;
	}

	/**
	 * This is the amount of dotlines, each 1/8mm, the paper will be 'backed-up'
	 * before printing is started. The default value is 0 dotlines.
	 * 
	 * @param dotLines The number of dotlines, each 1/8mm, to initially back-up.
	 *        Valid values are from 0 to 8192 dotlines.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void setInitialPaperBackup(int dotLines) throws IllegalArgumentException {

		// Check inputs
		if ((dotLines < 0) || (dotLines > 8192)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'dotLines' must be "
					+ "from 0 to 8192, a value of " + dotLines + " was given.");
		}

		// Assign Value
		m_InitialPaperBack = dotLines;

		return;
	}

	/**
	 * The page length is the maximum number of rows, in dot lines, that can be
	 * printed. This will put an upper bound on the document size if constraints
	 * require it. The default value is 0 meaning no maximum.
	 * 
	 * @return Maximum number of dotlines to print from the beginning of the
	 *         document.
	 * @since 1.0.0
	 */
	public int getPageLength() {
		return m_PageLength;
	}

	/**
	 * The page length is the maximum number of rows, in dot lines, that can be
	 * printed. This will put an upper bound on the document size if constraints
	 * require it. The default value is 0 meaning no maximum.
	 * 
	 * @param dotLines Maximum number of dotlines to print from the beginning of
	 *        the document. Valid values are from 0 to 8192 dotlines.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void setPageLength(int dotLines) throws IllegalArgumentException {

		// Check inputs
		if ((dotLines < 0) || (dotLines > 8192)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'dotLines' must be "
					+ "from 0 to 8192, a value of " + dotLines + " was given.");
		}

		// Assign Value
		m_PageLength = dotLines;

		return;
	}

	/**
	 * This is the number of copies of the document to print. The default number
	 * is 1.
	 * 
	 * @return The number of copies of the document to print.
	 * @since 1.0.0
	 */
	public int getPrintQuantity() {
		return m_PrintQuantity;
	}

	/**
	 * This will cause multiple print jobs to be 'optimized' by only performing
	 * the initial paper backup on the first item if set to true. Setting it to
	 * false will cause each job to be printed as if it were an individual
	 * request. The default value for this property is false.
	 * 
	 * @return If the print optimization mode is enabled.
	 * @since 1.0.0
	 */
	public boolean getPrintQuantityOptimized() {
		return m_PrintOptimize;
	}

	/**
	 * This sets the number of copies of the document to print. The default
	 * value for quantity is 1.
	 * 
	 * @param quantity The number of copies of the document to print. Valid
	 *        values are from 1 to 255.
	 * @throws IllegalArgumentException if the parameter value is not in the
	 *         valid range this exception is thrown.
	 * @since 1.1.0
	 */
	public void setPrintQuantity(int quantity) throws IllegalArgumentException {

		// Check inputs
		if ((quantity < 1) || (quantity > 255)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'quantity' must be "
					+ "from 1 to 255, a value of " + quantity + " was given.");
		}

		// Assign Value
		m_PrintQuantity = quantity;

		return;
	}

	/**
	 * This sets if the process should be optimized by removing the initial
	 * paper backup setting for the 2nd and greater copies. Setting optimize to
	 * true will cause multiple print jobs to be 'optimized' by only performing
	 * the initial paper backup on the first item if set to true. Setting it to
	 * false will cause each job to be printed as if it were an individual
	 * request. The default value for optimize is false.
	 * 
	 * @param optimize If the print optimizations mode is enabled.
	 * @since 1.1.0
	 */
	public void setPrintQuantityOptimized(boolean optimize) {

		// Assign Value
		m_PrintOptimize = optimize;

		return;
	}

	/**
	 * This gets the number of dotlines after the queue mark is found to stop
	 * feeding paper. This value is only used if QMarkStop is not the default
	 * value.  Each dotline is 1/8mm and the default value is 0.
	 * 
	 * @return The number of dotlines after the queue mark is found to stop.
	 * @since 1.0.0
	 */
	public int getQMarkDotLines() {
		return m_QMarkDotLines;
	}

	/**
	 * This gets the queue mark behavior for the printer. The default setting
	 * is QStop.Default causing the printer to act according to the value
	 * programmed into its internal flash memory. The other values available
	 * are:
	 * <UL>
	 * <LI>QStop.Default - The printer will act according to the value
	 * programmed into its internal flash memory.</LI>
	 * <LI>QStop.Front - The printer will stop printing the set number of
	 * dotlines after the queue mark is found on the <B>front</B> of the media.
	 * This mode applies to the LP3, MF2, MF3, 2t and 4t.</LI>
	 * <LI>QStop.Back - The printer will stop printing the set number of
	 * dotlines after the queue marks is found on the <B>back</B> of the media.
	 * This mode applies to the LP3.</LI>
	 * <LI>QStop.Gap - The printer will stop printing the set number of dotlines
	 * <B>after the inter-label gap</B> is found between labels. This mode
	 * applies to the LP3.</LI>
	 * <LI>QStop.None - The printer will ignore all queue stop settings in its
	 * internal flash memory and just print normally.</LI>
	 * </UL>
	 * Not all values are applicable to all printer models. Which ones apply
	 * depend on which sensors are installed on the particular printer.
	 * 
	 * @return The queue mark behavior for the printer.
	 * @since 1.0.0
	 */
	public QStop getQMarkStop() {
		return m_QMarkMode;
	}

	/**
	 * This sets the queue mark behavior for the printer. The default setting
	 * is QStop.Default causing the printer to act according to the value
	 * programmed into its internal flash memory. The other values available
	 * are:
	 * <UL>
	 * <LI>QStop.Default - The printer will act according to the value
	 * programmed into its internal flash memory.</LI>
	 * <LI>QStop.Front - The printer will stop printing the set number of
	 * dotlines after the queue mark is found on the <B>front</B> of the media.
	 * This mode applies to the LP3, MF2, MF3, 2t and 4t.</LI>
	 * <LI>QStop.Back - The printer will stop printing the set number of
	 * dotlines after the queue marks is found on the <B>back</B> of the media.
	 * This mode applies to the LP3.</LI>
	 * <LI>QStop.Gap - The printer will stop printing the set number of dotlines
	 * <B>after the inter-label gap</B> is found between labels. This mode
	 * applies to the LP3.</LI>
	 * <LI>QStop.None - The printer will ignore all queue stop settings in its
	 * internal flash memory and just print normally.</LI>
	 * </UL>
	 * Not all values are applicable to all printer models. Which ones apply
	 * depend on which sensors are installed on the particular printer.
	 * 
	 * @param mode The queue mark behavior for the printer.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.1.0
	 */
	public void setQMarkStop(QStop mode) throws IllegalArgumentException {

		// Check inputs
		if (mode == null) {
			throw new IllegalArgumentException("Parameter 'mode' cannot be null.");		
		}

		// Assign Value
		m_QMarkMode = mode;

		return;
	}

	/**
	 * This gets the number of dotlines after the queue mark is found to stop
	 * feeding paper. This value is only used if QMarkStop is not the default
	 * value.  Each dotline is 1/8mm and the default value is 0.
	 * 
	 * @param dotLines The number of dotlines after the queue mark is found
	 *        valid values are from 0 to 8192.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.1.0
	 */
	public void setQMarkDotLines(int dotLines) throws IllegalArgumentException {

		// Check inputs
		if ((dotLines < 0) || (dotLines > 8192)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'dotLines' must be "
					+ "from 0 to 8192, a value of " + dotLines + " was given.");
		}

		// Assign Value
		m_QMarkDotLines = dotLines;

		return;
	}

	/**
	 * This function will print the provided text to the document object using
	 * the default printing parameter values.
	 * 
	 * @param textString This is the text you wish to print.
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @throws IllegalArgumentException If any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeText(String textString, int row, int col) throws IllegalArgumentException {

		// Call full function
		writeText(textString, row, col, new ParametersEZ());

		return;
	}

	/**
	 * This function will print the provided text to the document object using
	 * the provided printing parameter values.
	 * 
	 * @param textString This is the text you wish to print.
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param parameters This ParametersEZ object specifies any printing
	 *        parameter values you wish to alter for the printing of this item.
	 * @throws IllegalArgumentException If any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeText(String textString, int row, int col, ParametersEZ parameters) throws IllegalArgumentException {

		// Validate position (throws on error)
		validatePosition(row, col);

		// Check inputs
		if (textString == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'textString' was null.");
		}

		if (parameters == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}

		// Add to Document
		byte[] paramData = parameterString(parameters, PrintingType.General);

		addToDoc(m_Document, "@" + row + "," + col + ":");
		addToDoc(m_Document, paramData);
		addToDoc(m_Document, "|" + textString + "|");
		addToDoc(m_Document, EOL);

		return;
	}

	/**
	 * This will print a horizontal line to the document at the given position
	 * with the given length and width.
	 * 
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param length Length of the line in dots
	 * @param thickness Thickness of the line in dots
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeHorizontalLine(int row, int col, int length, int thickness) throws IllegalArgumentException {

		// Call full function
		writeHorizontalLine(row, col, length, thickness, new ParametersEZ());

		return;
	}

	/**
	 * This will print a horizontal line to the document at the given position
	 * with the given length and width.
	 * 
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param length Length of the line in dots
	 * @param thickness Thickness of the line in dots
	 * @param parameters This ParametersEZ object specifies any printing
	 *        parameter values you wish to alter for the printing of this item.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeHorizontalLine(int row, int col, int length, int thickness, ParametersEZ parameters) throws IllegalArgumentException {
		int l = length;
		int t = thickness;

		// Validate position (throws on error)
		validatePosition(row, col);

		// Check inputs
		if ((length < 1) || (length > 8192)) {
			// Length range was invalid
			throw new IllegalArgumentException("Parameter 'length' must be "
					+ "an integer from 1 to 8192, a value of " + length + " was given.");
		}

		if ((thickness < 1) || (thickness > 8192)) {
			// Thickness range was invalid
			throw new IllegalArgumentException("Parameter 'thickness' must be "
					+ "an integer from 1 to 8192, a value of " + thickness + " was given.");
		}

		// Add to Document
		byte[] paramData = parameterString(parameters, PrintingType.Line);

		addToDoc(m_Document, "@" + row + "," + col);
		addToDoc(m_Document, ":HLINE,L" + l + ",T" + t);
		addToDoc(m_Document, paramData);
		addToDoc(m_Document, "|");
		addToDoc(m_Document, EOL);

		return;
	}

	/**
	 * This will print a vertical line to the document at the given position
	 * with the given length and width.
	 * 
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param length Length of the line in dots
	 * @param thickness Thickness of the line in dots
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeVerticalLine(int row, int col, int length, int thickness) throws IllegalArgumentException {

		// Call full function
		writeVerticalLine(row, col, length, thickness, new ParametersEZ());

		return;
	}

	/**
	 * This will print a vertical line to the document at the given position
	 * with the given length and width.
	 * 
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param length Length of the line in dots
	 * @param thickness Thickness of the line in dots
	 * @param parameters This ParametersEZ object specifies any printing
	 *        parameter values you wish to alter for the printing of this item.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeVerticalLine(int row, int col, int length, int thickness, ParametersEZ parameters) throws IllegalArgumentException {
		int l = length;
		int t = thickness;

		// Validate position (throws on error)
		validatePosition(row, col);

		// Check inputs
		if ((length < 1) || (length > 8192)) {
			// Length range was invalid
			throw new IllegalArgumentException("Parameter 'length' must be "
					+ "an integer from 1 to 8192, a value of " + length + " was given.");
		}

		if ((thickness < 1) || (thickness > 8192)) {
			// Thickness range was invalid
			throw new IllegalArgumentException("Parameter 'thickness' must be "
					+ "an integer from 1 to 8192, a value of " + thickness + " was given.");
		}

		// Add to Document
		byte[] paramData = parameterString(parameters, PrintingType.Line);

		addToDoc(m_Document, "@" + row + "," + col);
		addToDoc(m_Document, ":VLINE,L" + l + ",T" + t);
		addToDoc(m_Document, paramData);
		addToDoc(m_Document, "|");
		addToDoc(m_Document, EOL);

		return;
	}

	/**
	 * This function will print the provided text to the document as a barcode
	 * using the font specified and the default printing parameter values.
	 * 
	 * @param bcFont Name of the barcode font.
	 * @param textString Text to print as barcode.
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeBarCode(String bcFont, String textString, int row, int col) throws IllegalArgumentException {
		ParametersEZ paramEZ = new ParametersEZ();
		
		// Set the barcode default sizes to something "normal" for the people who
		// can't read the help docs.
		paramEZ.setHorizontalMultiplier(2);
		paramEZ.setVerticalMultiplier(10);
		
		// Call full function
		writeBarCode(bcFont, textString, row, col, paramEZ);

		return;
	}

	/**
	 * This function will print the provided text to the document as a barcode
	 * using the font specified and the provided printing parameter values.
	 * 
	 * @param bcFont Name of the barcode font.
	 * @param textString Text to print as barcode.
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param parameters This ParametersEZ object specifies any printing
	 *        parameter values you wish to alter for the printing of this item.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeBarCode(String bcFont, String textString, int row, int col, ParametersEZ parameters) throws IllegalArgumentException {

		// Validate position (throws on error)
		validatePosition(row, col);

		// Check inputs
		if ((bcFont == null) || (bcFont.length() != m_FontNameLength)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'bcFont' must be " + m_FontNameLength
					+ " characters in length.");
		}

		if (textString == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'textString' was null.");
		}

		if (parameters == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}

		// Add to Document
		byte[] paramData = parameterString(parameters, PrintingType.Barcode);

		addToDoc(m_Document, "@" + row + "," + col + ":" + bcFont);
		addToDoc(m_Document, paramData);
		addToDoc(m_Document, "|" + textString + "|");
		addToDoc(m_Document, EOL);

		return;
	}

	/**
	 * This function will print the provided text to the document as a PDF417 2D
	 * barcode using the default printing parameter values.
	 * 
	 * @param textString Text to print as PDF417 2D barcode.
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param dataColumns Use this parameter to override the default of 2
	 *        columns for the PDF-417 bar code to specify the actual number of
	 *        data columns being printed across any given line of the bar code.
	 * @param redundancyLevel This will set the number of redundant codewords
	 *        used to protect against data loss/corruption. The typical value is
	 *        1 which would result in 4 codewords. Other values are 2 giving you
	 *        8 codewords, 3 giving you 16 codewords up to 8 giving you 512
	 *        codewords.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeBarCodePDF417(String textString, int row, int col, int dataColumns, int redundancyLevel) throws IllegalArgumentException {
		ParametersEZ paramEZ = new ParametersEZ();
		
		// Set the barcode default sizes to something "normal" for the people who
		// can't read the help docs.
		paramEZ.setHorizontalMultiplier(2);
		paramEZ.setVerticalMultiplier(10);
		
		// Call full function
		writeBarCodePDF417(textString, row, col, dataColumns, redundancyLevel, paramEZ);

		return;
	}

	/**
	 * This function will print the provided text to the document as a PDF417 2D
	 * barcode using the provided printing parameter values.
	 * 
	 * @param textString Text to print as PDF417 2D barcode.
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param dataColumns Use this parameter to override the default of 2
	 *        columns for the PDF-417 bar code to specify the actual number of
	 *        data columns being printed across any given line of the bar code.
	 * @param redundancyLevel This will set the number of redundant codewords
	 *        used to protect against data loss/corruption. The typical value is
	 *        1 which would result in 4 codewords. Other values are 2 giving you
	 *        8 codewords, 3 giving you 16 codewords up to 8 giving you 512
	 *        codewords.
	 * @param parameters This ParametersEZ object specifies any printing
	 *        parameter values you wish to alter for the printing of this item.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeBarCodePDF417(String textString, int row, int col, int dataColumns, int redundancyLevel, ParametersEZ parameters) throws IllegalArgumentException {

		// Validate position (throws on error)
		validatePosition(row, col);

		// Check inputs
		if (textString == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'textString' was null.");
		}

		if ((dataColumns < 1) || (dataColumns > 255)) {
			// Data columns must be a greater than zero
			throw new IllegalArgumentException("Parameter 'dataColumns' must be a "
					+ "positive integer from 1 to 255, a value of " + dataColumns + " was given.");
		}

		if ((redundancyLevel < 1) || (redundancyLevel > 8)) {
			// Redundancy is from 1 to 8
			throw new IllegalArgumentException("Parameter 'redundancyLevel' must "
					+ "be an integer from 1 to 8, a value of " + redundancyLevel + " was given.");
		}

		if (parameters == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}

		// Add to Document
		byte[] paramData = parameterString(parameters, PrintingType.Barcode417);

		addToDoc(m_Document, "@" + row + "," + col);
		addToDoc(m_Document, ":PD417,COLUMNS" + dataColumns + ",SECURITY" + redundancyLevel);
		addToDoc(m_Document, paramData);
		addToDoc(m_Document, "|");
		addToDoc(m_Document, textString);
		addToDoc(m_Document, "|");
		addToDoc(m_Document, EOL);

		return;
	}

	/**
	 * This will cause the image specified, which is stored on the printer, to
	 * be printed at the given location.
	 * 
	 * @param imageName Name of the image stored in printer memory.
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeImage(String imageName, int row, int col) throws IllegalArgumentException {

		// Call full function
		writeImage(imageName, row, col, new ParametersEZ());

		return;
	}

	/**
	 * This will cause the image specified, which is stored on the printer, to
	 * be printed at the given location.
	 * 
	 * @param imageName Name of the image stored in printer memory.
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param parameters This ParametersEZ object specifies any printing
	 *        parameter values you wish to alter for the printing of this item.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	public void writeImage(String imageName, int row, int col, ParametersEZ parameters) throws IllegalArgumentException {

		// Validate position (throws on error)
		validatePosition(row, col);

		// Check inputs
		if (imageName == null) {
			// Invalid image name object
			throw new IllegalArgumentException("Parameter 'imageName' was null.");
		}

		if (imageName.length() == 0) {
			// Invalid image name object
			throw new IllegalArgumentException("Parameter 'imageName' "
					+ "can not be the empty string.");
		}

		if (parameters == null) {
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}

		// Add to Document
		byte[] paramData = parameterString(parameters, PrintingType.Image);

		addToDoc(m_Document, "@" + row + "," + col + ":" + imageName);
		addToDoc(m_Document, paramData);
		addToDoc(m_Document, "|");
		addToDoc(m_Document, EOL);

		return;
	}
	
	/**
	 * This will print a horizontal line to the document at the given position
	 * with the given length and width.
	 * 
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param width Width of the area in dots.
	 * @param height Height of the area in dots.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 */
	public void writeInverseRegion(int row, int col, int width, int height) throws IllegalArgumentException {

		// Call full function
		writeInverseRegion(row, col, width, height, new ParametersEZ());

		return;
	}

	/**
	 * This will print a horizontal line to the document at the given position
	 * with the given length and width.
	 * 
	 * @param row Row position, starting from zero, to start printing at.
	 * @param col Column position, starting from zero, to start printing at.
	 * @param width Width of the area in dots.
	 * @param height Height of the area in dots.
	 * @param parameters This ParametersEZ object specifies any printing
	 *        parameter values you wish to alter for the printing of this item.
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 */
	public void writeInverseRegion(int row, int col, int width, int height, ParametersEZ parameters) throws IllegalArgumentException {
		int l = width;
		int t = height;

		// Validate position (throws on error)
		validatePosition(row, col);

		// Check inputs
		if ((width < 1) || (width > 8192)) {
			// Length range was invalid
			throw new IllegalArgumentException("Parameter 'width' must be "
					+ "an integer from 1 to 8192, a value of " + width + " was given.");
		}

		if ((height < 1) || (height > 8192)) {
			// Thickness range was invalid
			throw new IllegalArgumentException("Parameter 'height' must be "
					+ "an integer from 1 to 8192, a value of " + height + " was given.");
		}

		// Add to Document
		byte[] paramData = parameterString(parameters, PrintingType.Line);

		addToDoc(m_Document, "@" + row + "," + col);
		addToDoc(m_Document, ":INVRS,L" + l + ",T" + t);
		addToDoc(m_Document, paramData);
		addToDoc(m_Document, "|");
		addToDoc(m_Document, EOL);

		return;
	}

	/**
	 * We check the row and column values in so many functions that changes to
	 * the validation would be cumbersome. So I centralized it here to make it
	 * easier to maintain.
	 * 
	 * @param row Value for the starting row position
	 * @param col Value for the starting column position
	 * @throws IllegalArgumentException if any of the parameter values are not
	 *         in the valid range this exception is thrown.
	 * @since 1.0.0
	 */
	private void validatePosition(int row, int col) throws IllegalArgumentException {

		if ((row < 0) || (row > 8192)) {
			// Row range was invalid
			throw new IllegalArgumentException("Parameter 'row' must be "
					+ "an integer from 0 to 8192, a value of " + row + " was given.");
		}

		if ((col < 0) || (col > 8192)) {
			// Column range was invalid
			throw new IllegalArgumentException("Parameter 'col' must be "
					+ "an integer from 0 to 8192, a value of " + col + " was given.");
		}

		return;
	}

	/**
	 * This will build the parameters string for the current type of object
	 * being rendered.
	 * 
	 * @param parameters Parameter object to get settings from.
	 * @param type Type of object being rendered.
	 * @return The parameters encoded into the printer's language for the
	 *         current object.
	 */
	private byte[] parameterString(ParametersEZ parameters, PrintingType type) {
		ByteArrayOutputStream paramData = new ByteArrayOutputStream(128);

		// Set Font
		if (type == PrintingType.General) {
			// Font name only applies to this type
			if (parameters.getFont() == "")
				addToDoc(paramData, getDefaultFont());
			else
				addToDoc(paramData, parameters.getFont());
		}

		// Inverse Printing
		if ((type == PrintingType.General) || (type == PrintingType.Line) || (type == PrintingType.Image)) {
			if (parameters.getIsInverse())
				addToDoc(paramData, ",I");
		}

		// Field Rotation
		if (parameters.getRotate() == Rotation.Rotate_90) {
			addToDoc(paramData, ",ROT90");
		}
		else if (parameters.getRotate() == Rotation.Rotate_180) {
			addToDoc(paramData, ",ROT180");
		}
		else if (parameters.getRotate() == Rotation.Rotate_270) {
			addToDoc(paramData, ",ROT270");
		}
	
		// Alignment
		if (parameters.getAlignment() == Alignment.Center) {
			addToDoc(paramData, ",CENTER");
		}
		else if (parameters.getAlignment() == Alignment.Right) {
			addToDoc(paramData, ",RIGHT");			
		}

		// Type Specific
		if ((type == PrintingType.General) || (type == PrintingType.Line) || (type == PrintingType.Image)) {
			// General Fields
			if (parameters.getHorizontalMultiplier() != 1) {
				addToDoc(paramData, ",HM" + parameters.getHorizontalMultiplier());
			}

			if (parameters.getVerticalMultiplier() != 1) {
				addToDoc(paramData, ",VM" + parameters.getVerticalMultiplier());
			}
		}
		else if (type == PrintingType.Barcode) {
			// Standard Bar Codes
			if (parameters.getHorizontalMultiplier() != 1) {
				addToDoc(paramData, ",W" + parameters.getHorizontalMultiplier());
			}

			if (parameters.getVerticalMultiplier() != 1) {
				addToDoc(paramData, ",H" + parameters.getVerticalMultiplier());
			}
		}
		else if (type == PrintingType.Barcode417) {
			// PDF417 Bar Codes (The default is 3 so always state)
			addToDoc(paramData, ",XDIM" + parameters.getHorizontalMultiplier());
			addToDoc(paramData, ",YDIM" + parameters.getVerticalMultiplier());
		}

		return paramData.toByteArray();
	}

	/**
	 * This method will return the rendered document in the printer's language
	 * for sending directly to the printer.
	 * 
	 * @return Rendered document in the printer's language.
	 * @since 1.1.0
	 */
	@Override
	public byte[] getDocumentData() {
		String globals = "";
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();

		// Form global parameter settings
		if (getInitialPaperBackup() > 0) {
			// Backup n dotlines
			globals += ",BACK" + getInitialPaperBackup();
		}

		if (getQMarkStop() == QStop.Front) {
			// Mark at the front
			globals += ",QSTOPT" + getQMarkDotLines();
		}
		else if (getQMarkStop() == QStop.Back) {
			// Mark on the back of the paper
			globals += ",QSTOPB" + getQMarkDotLines();
		}
		else if (getQMarkStop() == QStop.Gap) {
			// Between label gap
			globals += ",QSTOPG" + getQMarkDotLines();
		}
		else if (getQMarkStop() == QStop.None) {
			// Disable
			globals += ",QSTOPN";
		}

		if ((getPrintQuantity() != 1) && (getPrintQuantityOptimized() == true)) {
			// Optimized Printing
			globals += ",QUANTITY" + getPrintQuantity();
		}
		else if ((getPrintQuantity() != 1) && (getPrintQuantityOptimized() == false)) {
			// Non-optimized printing
			globals += ",QUANTITYNOPT" + getPrintQuantity();
		}

		if (getIsLandscapeMode()) {
			// Use landscape mode
			globals += ",ROT270";
		}

		if (getPageLength() != 0) {
			// Put a limit on the printable length
			globals += ",STOP" + getPageLength();
		}

		// Build
		addToDoc(dataStream, ESC + "EZ{PRINT" + globals + ":\r\n");
		addToDoc(dataStream, m_Document.toByteArray());
		addToDoc(dataStream, "}\r\n");

		return dataStream.toByteArray();
	}
}
