package datamaxoneil.printer;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;

import javax.imageio.ImageIO;

/**
 * This class handles the interface between the printer and the user application
 * that wishes to use a Datamax-O'Neil Apex printer.
 * A simple example is:<BR>
 * <BR>
 * <CODE><PRE>
 *    // Print Image
 *    DocumentEXPCL docEXPCL;
 *    docEXPCL = new DocumentEXPCL();
 *    //Specify which model printer we want to print to.
 *    docEXPCL.setPrinterModel(PrinterModel.APEX3)
 *    //Write data to datastream
 *    docEXPCL.writeImage("/sdcard/picture.png");
 * </PRE></CODE> This document can then be sent to a PrinterONeil object through
 * the print method to print out.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class DocumentEXPCL extends Document {

	/**
	 * Apex Printer Models
	 *
	 */
	public enum PrinterModel
	{
		//2-inch Printer
		APEX2,
		//3-inch
		APEX3,
		//4-inch
		APEX4
	}
	
	private PrinterModel m_printerModel = PrinterModel.APEX3;
	
	/**
	 * This method prints the image specified by the user.
	 * @param fileName - The file path of the image file to be printed.
	 */
	public void writeImage(String fileName)
	{
		
		try {
			
			//Open file and store picture data in byte array
			BufferedImage imageObject = ImageIO.read(new File(fileName));
			
			//Get store image data in byte array
			int height = imageObject.getHeight();
			int width = imageObject.getWidth();
			
			//Get printHeadWidth of printer base on model
			int printHeadWidth = 0;
			PrinterModel printerModel = getPrinterModel();
			switch(printerModel)
			{
				case APEX2:
					printHeadWidth = 384;
					break;
				case APEX3:
					printHeadWidth = 576;
					break;
				case APEX4:
					printHeadWidth = 832;
					break;
				
			}
			//Tells printer to send down graphic/image. Format: <ESC>Vn1n2
			// where n1n2 is the bitmap height.
			// n1: least significant byte
			// n2: most significant byte
			byte[] graphicCommand = {(byte)ESC, 'V', (byte)height, 0x00};   //the bitmap height in this sample is 0x00F6
			
			addToDoc(m_Document, graphicCommand);

			//array for each graphic line. size of array is converted into bytes(eg. 576 pixels = 72 bytes per line)
			byte[] dataline = new byte[(printHeadWidth + 7) >> 3];
			int[] imageData = new int[height * width];
			/*Android version*/
			//imageObject.get(imageData, 0, width, 0, 0, width, height);
			/*Java version*/
			imageObject.getRGB(0, 0, width, height, imageData, 0, width);
			// Loop through each row
			for (int row = 0; row < height; row++) {
				//boolean blankLine = true;
				// Get the current scanline
				for (int index = 0; index < width; index += 8) {
					int value;
					boolean set;
					byte currentByte = 0;
					int offset = (row * width) + index;

					// Check for EOL
					if (index >= printHeadWidth)
						break;

					value = ((index + 0) < width) ? imageData[offset + 0] & 0x00FFFFFF : 0x00FFFFFF;
					set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
					currentByte |= (set) ? (byte) 0x80 : (byte) 0x00;

					value = ((index + 1) < width) ? imageData[offset + 1] & 0x00FFFFFF : 0x00FFFFFF;
					set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
					currentByte |= (set) ? (byte) 0x40 : (byte) 0x00;

					value = ((index + 2) < width) ? imageData[offset + 2] & 0x00FFFFFF : 0x00FFFFFF;
					set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
					currentByte |= (set) ? (byte) 0x20 : (byte) 0x00;

					value = ((index + 3) < width) ? imageData[offset + 3] & 0x00FFFFFF : 0x00FFFFFF;
					set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
					currentByte |= (set) ? (byte) 0x10 : (byte) 0x00;

					value = ((index + 4) < width) ? imageData[offset + 4] & 0x00FFFFFF : 0x00FFFFFF;
					set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
					currentByte |= (set) ? (byte) 0x08 : (byte) 0x00;

					value = ((index + 5) < width) ? imageData[offset + 5] & 0x00FFFFFF : 0x00FFFFFF;
					set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
					currentByte |= (set) ? (byte) 0x04 : (byte) 0x00;

					value = ((index + 6) < width) ? imageData[offset + 6] & 0x00FFFFFF : 0x00FFFFFF;
					set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
					currentByte |= (set) ? (byte) 0x02 : (byte) 0x00;

					value = ((index + 7) < width) ? imageData[offset + 7] & 0x00FFFFFF : 0x00FFFFFF;
					set = ((((value >> 0) & 0xFF) + ((value >> 8) & 0xFF) + ((value >> 16) & 0xFF)) < (128 * 3));
					currentByte |= (set) ? (byte) 0x01 : (byte) 0x00;

					// Now set in dataline
					dataline[index >> 3] = currentByte;
					//blankLine &= (currentByte == 0x00);
				}

					// Compress it and add to document
					addToDoc(m_Document, dataline);  
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
               

	}

	@Override
	public byte[] getDocumentData() {
		// TODO Auto-generated method stub
		ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
		addToDoc(dataStream, m_Document.toByteArray());

		return dataStream.toByteArray();
	}

	/**
	 * Gets the printerModel of DocumentEXPCL
	 * 
	 */
	public PrinterModel getPrinterModel() {
		return m_printerModel;
	}

	/**
	 * Sets the printerModel of DocumentEXPCL
	 * @param printerModel - printer model to be set
	 */
	public void setPrinterModel(PrinterModel printerModel) {
		m_printerModel = printerModel;
	}

}
