package datamaxoneil.printer;


import java.awt.image.BufferedImage;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;

import javax.imageio.ImageIO;

import datamaxoneil.printer.ParametersDPL.Alignment;
import datamaxoneil.printer.ParametersDPL.Rotation;



/** 
 This class handles the interface between the printer and the user 
 application that wishes to use an Datamax-O'Neil DPL printer. 
 
 
 These functions can be used to 'build' a document using abstract methods
 instead of the raw 'DPL' mode commands. The DocumentEZ class is
 designed when exact placement of items is desired.
 
*/
public class DocumentDPL extends Document
{
	/** 
	 The type of format operation for each label in DPL mode.
	*/
	public enum Label_Format
	{
		/** 
		 Image and bar code intersect will not be printed
		*/
		XOR_Mode(1),
		/** 
		 intersecting region of text string will print
		*/
		Transparent_Mode(2),
		/** 
		 intersecting region of text string will not be print
		*/
		Opaque_Mode(3),
		/** 
		 inverse printing.
		*/
		Inverse_Mode(5);

		private int value;

		private Label_Format(int intValue)
		{
			value = intValue;
		}

		public int getValue()
		{
			return value;
		}
	}
	/** 
	 Image Type to be printed
	*/
	public enum ImageType
	{
		/** 
		 7-Bit D-O image load file
		*/
		DOImage_7Bit,

		/** 
		 .BMP 8-bit format, flipped, Black and White
		*/
		BMPFlipped_8Bit,

		/** 
		 .BMP 8-bit format, Black and White
		*/
		BMP_8Bit,

		/** 
		 .IMG 8-bit format, Black and White
		*/
		IMG_8Bit,

		/** 
		 .IMG 8-bit format, flipped, Black and White
		*/
		IMGFlipped_8Bit,

		/** 
		 .PCX 8-bit format, flipped, Black and White
		*/
		PCXFlipped_8Bit,

		/** 
		 .PCX 8-bit format, flipped, Black and White
		*/
		PCX_8Bit,

		/** 
		 Unknown format
		*/
		Other;
	}

	/** 
	 Font Type of text to be printed
	*/
	public enum Font_Type
	{
		/** 
		 Internal Bitmapped Fonts of printer
		*/
		InternalBitmapped(0),

		/** 
		 Scalable Fonts
		*/
		Scalable(9);

		private int value;
	
		private Font_Type(int intValue)
		{
			value = intValue;
		}

		public int getValue()
		{
			return value;
		}

	}
	/** 
	 GS Hex value = 1D 
	*/
	public static char GS = (char)(29);
	/** 
	 RS Hex Value = 1E
	*/
	public static char RS = (char)(30);
	/** 
	 EOT Hex value = 04
	*/
	public static char EOT = (char)(04);
			/** 
	 Vertical adjustment of the point where printing begins
			*/
	private int m_RowOffset = 0;
	/** 
	 Horizontal adjustment of the point where printing begins
	*/
	private int m_ColumnOffset = 0;

	/** 
	 The dot width multiplier of a printed dot.
	 Valid values - 1 or 2
	*/
	private int m_DotWidthMultiplier = 1;

	/** 
	 The dot height multiplier of printed dot
	 Valid values - 1, 2, or 3
	*/
	private int m_DotHeightMultiplier = 1;

	/** 
	The "on time" of elements of the print head. The default setting is
	10
	*/
	private int m_HeatSettings = 10;

	/** 
	 The number of label copies to be printed
	*/
	private int m_PrintQuantity;

	/** 
	 The type of format operation for label to be print
	*/
	private Label_Format m_LabelFormat = Label_Format.XOR_Mode;

	/** 
	 The type of format operation for label to be print
	*/
	public Label_Format getLabelFormat()
	{
		return m_LabelFormat;
	}
	

	/**
	 * Set the label format
	 * @param value - label format value
	 */
	public void setLabelFormat(Label_Format value)
	{
		m_LabelFormat = value;
	}

	/** 
	Gets the column offset
	*/
	public int getColumnOffset()
	{
		return m_ColumnOffset;
	}
	/**
	 * Set the column offset
	 * @param offset - offset value
	 */
	public void setColumnOffset(int offset)
	{
		if ((offset < 0) || (offset >9999))
		{
				// Out of range
			throw new IllegalArgumentException(String.format("Parameter 'columnOffset' must be from 0 - 9999, a value of %1$s was given.", offset));
		}
		m_ColumnOffset = offset;
	}

	/** 
	 Gets the Row offset
	*/
	public int getRowOffset()
	{
		return m_RowOffset;
	}
	/**
	 * Set the row offset
	 * @param value - offset value
	 */
	public void setRowOffset(int value)
	{
		if ((value < 0) || (value >9999))
		{
				// Out of range
			throw new IllegalArgumentException(String.format("Parameter 'rowOffset' must be from 0 - 9999, a value of %1$s was given.", value));
		}
		m_RowOffset = value;
	}

	/** 
	 Gets the Dot width multiplier
	*/
	public int getDotWidthMultiplier()
	{
		return m_DotWidthMultiplier;
	}
	/**
	 * Set dot width multiplier
	 * @param value - dot width multiplier value
	 */
	public void setDotWidthMultiplier(int value)
	{ // Check inputs
		if ((value < 1) || (value > 2))
		{
				// Out of range
			throw new IllegalArgumentException(String.format("Parameter 'dot-width-multiplier' must be from 1 to 2, a value of %1$s was given.", value));
		}
	}

	/** 
	 Gets the dot height multiplier
	*/
	public int getDotHeightMultiplier()
	{
		return m_DotHeightMultiplier;
	}
	/**
	 * Sets the dot height multiplier
	 * @param value - dot height multiplier value
	 */
	public void setDotHeightMultiplier(int value)
	{
			// Check inputs
		if ((value < 1) || (value > 3))
		{
				// Out of range
			throw new IllegalArgumentException(String.format("Parameter 'dot-height-multiplier' must be from 1 to 3, a value of %1$s was given.", value));
		}
			//Assign value
		m_DotHeightMultiplier = value;
	}

	/** 
	 Gets the heat settings of document
	*/
	public int getHeatSettings()
	{
		return m_HeatSettings;
	}
	/**
	 * Sets the heat settings of document
	 * @param value - heat settings value
	 */
	public void setHeatSettings(int value)
	{
			//Checks inputs
		if ((value < 0) || (value > 30))
		{
				// Out of range
			throw new IllegalArgumentException(String.format("Parameter 'heat settings' must be from 0 to 30, a value of %1$s was given.", value));
		}
		m_HeatSettings = value;
	}


	/** 
	 Gets the number of labels copies to be printed
	*/
	public int getPrintQuantity()
	{
		return m_PrintQuantity;
	}
	/**
	 * Sets the number of label copies to be printed
	 * @param value - copies to print
	 */
	public void setPrintQuantity(int value)
	{
			// Check inputs
		if ((value < 1) || (value > 255))
		{
				// Out of range
			throw new IllegalArgumentException(String.format("Parameter 'quantity' must be from 1 to 255, a value of %1$s was given.", value));
		}
		m_PrintQuantity = value;
	}

	/** 
	 This function will print the provided text to the document object using Internal Bitmapped
	 Fonts. Set the internal font id type to the 
	 
	 @param textString - Text to be printed
	 @param row -Row to start printing. Valid values 0 - 9999
	 @param col - col to start printing. Valid values 0-9999
	 @param fontID - Internal font ID type to . Valid values 0 - 8
	*/
	public void writeTextInternalBitmapped(String textString, int fontID, int row, int col)
	{
		ParametersDPL parameters = new ParametersDPL();
		parameters.setFont("000");

		//Validate font ID value
		if (fontID < 0 || fontID > 8)
		{
			throw new IllegalArgumentException(String.format("Parameter 'font-id' must be between 0 - 8. Value given was %1$s", fontID));
		}
		parameters.setTypeID(Integer.toString(fontID));

		writeText(textString, row, col, parameters.getTypeID(), Font_Type.InternalBitmapped, parameters);
		return;
	}

	/** 
	 This function will print the provided text to the document object 
	 using Scalable Font .
	 
	 @param textString - This is the text you wish to print.
	 @param row - Row position, starting from zero, to start printing 
	 at.
	 @param col - Column position, starting from zero, to start
	 printing at.
	 @param fontID - Font ID of the Scalable Font
	*/
	public void writeTextScalable(String textString, String fontID, int row, int col)
	{
		ParametersDPL parameters = new ParametersDPL();

		//Setting reasonable parameters for printing
		parameters.setIsDotMode(false);
		parameters.setIsUnicode(false);
		parameters.setFontHeight(12);
		parameters.setFontWidth(12);

		writeTextScalable(textString, fontID, row, col, parameters);
		return;
	}

	/** 
	 This function will print the provided text to the document object 
	 using Scalable Font .
	 
	 @param textString - This is the text you wish to print.
	 @param row Row - position, starting from zero, to start printing 
	 at.
	 @param col - Column position, starting from zero, to start
	 printing at.
	 @param fontID - FontID of scalable Font ID
	 @param parameters - ParametersDPL object
	*/
	public void writeTextScalable(String textString, String fontID, int row, int col, ParametersDPL parameters)
	{
		parameters.setFont(fontID);
		//Checks if font height/width is used
		if (parameters.getFontHeight() != 0 && parameters.getFontWidth() != 0)
		{
			//Checks if we are using point units for font height/width
			if (!parameters.getIsDotMode())
			{

				if (parameters.getFontHeight() < 4 && parameters.getFontHeight()> 999)
				{
					throw new IllegalArgumentException(String.format("Parameter 'fontHeight' must be between 4 - 999 for points unit. Value given was %1$s", parameters.getFontHeight()));
				}
				if (parameters.getFontWidth() < 4 && parameters.getFontWidth() > 999)
				{
					throw new IllegalArgumentException(String.format("Parameter 'fontWidth' must be between 4 - 999 for points unit. Value given was %1$s", parameters.getFontWidth()));
				}


				//Format the text string to PhhhPjjjiiiii(eg. P004P004Hello)
				textString = "P" + String.format("%03d", parameters.getFontHeight()) + "P" + String.format("%03d", parameters.getFontWidth()) + textString;
			}
			//else if are using dots
			else
			{
				//Validate parameters
				if (parameters.getFontHeight() < 14 && parameters.getFontHeight()> 4163)
				{
					throw new IllegalArgumentException(String.format("Parameter 'fontHeight' must be between 14 - 4163 for points unit. Value given was %1$s", parameters.getFontHeight()));
				}
				if (parameters.getFontWidth() < 16 && parameters.getFontWidth() > 4163)
				{
					throw new IllegalArgumentException(String.format("Parameter 'fontWidth' must be between 16 - 4163 for points unit. Value given was %1$s", parameters.getFontWidth()));
				}

				//Format the text string to hhhhjjjjiiiii
				textString = String.format("%04d", parameters.getFontHeight()) + String.format("%04d", parameters.getFontWidth()) + textString;
			}
		}
		// Call full function

		writeText(textString, row, col, "9", Font_Type.Scalable, parameters);
		return;
	}

	/** 
	 This function will print the provided text to the document object 
	 using the provided printing parameter values.
	 
	 @param textString - This is the text you wish to print.
	 @param row - Row position, starting from zero, to start printing 
	 at.
	 @param col - Column position, starting from zero, to start printing
	 at.
	 @param fontIDType - Font ID type. 
	 Valid values:  0 to 8 for Internal Bitmap Fonts, 9 for Scalable and Smooth Fonts 
	 @param fontType - Type of font for print job
	 @param parameters - This ParametersDPL object specifies any additional
	 printing settings
	*/
	public void writeText(String textString, int row, int col, String fontIDType, Font_Type fontType, ParametersDPL parameters)
	{
		
		// Validate position (throws on error)
		ValidatePosition(row, col);

		// Check inputs
		if (textString == null)
		{
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'textString' was null.");
		}

		if (parameters == null)
		{
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}

		// Add to Document
		byte[] tempArray;

		//Get parameter string
		tempArray = parameterString(fontIDType, parameters, PrintingType.General);
		addToDoc(m_Document,tempArray);

		//write row and col postion
		addToDoc(m_Document,String.format("%04d", row) + String.format("%04d", row));

		addToDoc(m_Document,textString);

		addToDoc(m_Document, EOL);

		return;
	}

	/** 
	 This function will print the provided text to the document as a
	 barcode using the font specified and the default printing parameter
	 values.
	 
	 @param barcodeID ID of the barcode font. Valid values: A to Z(bc with human readable text), a to z (non-human readable text)(except P,u,v,z)
	 or Wna(n = 1-9, a = a-s/A-S)
	 @param textString Text to print as barcode.
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	*/
	public void writeBarCode(String barcodeID, String textString, int row, int col)
	{
		ParametersDPL parameters = new ParametersDPL();

		// Set the barcode default sizes to something "normal" for the people who
		// can't read the help docs.
		parameters.setWideBarWidth(0);
		parameters.setNarrowBarWidth(0);
		//Use default
		parameters.setSymbolHeight(0);

		// Call full function
		writeBarCode(barcodeID, textString, row, col, parameters);

		return;
	}

	/** 
	 This function will print the UPS MaxiCode bar codes.
	 
	 @param mode Maxicode Mode.Valid values: 2 = US locations, 3 = International locations
	 @param message UPS message for barcode.
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeBarCodeUPSMaxiCode(int mode, UPSMessage message, int row, int col, ParametersDPL parameters)
	{
		//Validate mode 
		if (mode != 2 && mode != 3)
		{
			throw new IllegalArgumentException("Parameter 'mode' must be either 2 or 3. Value given was " + mode);
		}


		//Start UPS message
		String textString = "#" + Integer.valueOf(mode).toString() + "[)>" + RS + "01" + GS + "96";

		//write zipcode
		//for US addressses
		if (mode == 2)
		{
			//Validate Zip code
			if (message.getZipCode().length()!= 5 && message.getZipCode().length() != 9)
			{
				throw new RuntimeException("Zip code provided for Mode 2 (US) must be 5 or 9 digits.");
			}
			//write ZIP code
			textString += rightPad(message.getZipCode(),9,'0') + GS;
			if (message.getCountryCode() != 840)
			{
				throw new RuntimeException("Country Code provided for Mode 2 (US) must be 840");
			}
			textString += String.format("%03d", message.getCountryCode())+ GS;
		}
		else if (mode == 3)
		{
			//Validate Zip code
			if (message.getZipCode().length() != 6)
			{
				throw new RuntimeException("Zip code provided for Mode 3 (International) must be 6 characters in length.");
			}
			//write international ZIP code
			textString += message.getZipCode() + GS;

			if (message.getCountryCode() == 840)
			{
				throw new RuntimeException("Country Code provided for Mode 3 (International) cannot be 840");
			}
			textString += String.format("%03d", message.getCountryCode()) + GS;
		}
		textString += String.format("%03d", message.getClassOfService()) + GS + message.getTrackingNumber() + GS + message.getSCAC() + GS + message.getShipperNumber() + GS + String.format("%03d", message.getJulianPickupDay()) + 
				GS + message.getShipmentID() + GS + Integer.valueOf(message.getCurrentPackage()).toString() + "/" + Integer.valueOf(message.getTotalPackage()).toString() + GS + Integer.valueOf(message.getPackageWeight()).toString() + 
				GS + ((message.getValidateAddress() == true) ? "Y" : "N") + GS + message.getShipToAddr() + GS + message.getShipToCity() + GS + message.getShipToState() + RS + EOT;
		writeBarCode("u", textString, row, col, parameters);


		return;
	}
	/** 
	 This function will print the provided text to the document as a
	 PDF417 2D barcode using the provided printing parameter values.
	 
	 @param textString Data to be encoded
	 @param isTruncated Specifies a normal or truncated bar code
	 @param securityLevel 1-digit security level ranging from 0 to 8
	 @param aspectRatio 2-digit aspect ratio specified as a fraction, with first digit being numerator and second digit the denominator. 
	 Use "00" for default ratio of 1:2. Valid range is from "00" to "99".
	 @param dataRows 2-digit number specifying number of rows requested. Valid range 3-90. Default value = 0
	 to let printer find best fit
	 @param dataCols 2-digit number specifying number of cols requested. Valid range 1-30. Default value = 0
	 to let printer find best fit
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeBarCodePDF417(String textString, boolean isTruncated, int securityLevel, int aspectRatio, int dataRows, int dataCols, int row, int col, ParametersDPL parameters)
	{
		//Validate security lvl
		if (securityLevel < 0 || securityLevel > 8)
		{
			throw new IllegalArgumentException("Parameter 'securityLevel' must be between 0-8. Value given was " + Integer.valueOf(securityLevel).toString());
		}
		//validate aspect ratio
		if (aspectRatio < 0 || aspectRatio > 99)
		{
			throw new IllegalArgumentException("Parameter 'aspectRatio' must be between 0-99. Value given was " + Integer.valueOf(aspectRatio).toString());
		}
		//Validate data rows
		if (dataRows < 0 || dataRows > 99)
		{
			throw new IllegalArgumentException("Parameter 'dataRows' must be between 0-99. Value given was " + Integer.valueOf(dataRows).toString());
		}

		//Validate data Cols
		if (dataCols < 0 || dataCols > 99)
		{
			throw new IllegalArgumentException("Parameter 'dataCols' must be between 0-99. Value given was " + Integer.valueOf(dataCols).toString());
		}
		//Start textstring
		String newTextString = ((isTruncated) ? "T" : "F") + Integer.valueOf(securityLevel).toString() + String.format("%02d",aspectRatio);

		//Add data rows. If value is >90, set to 90, value < 3 set to 3, if value = 0 set to 0.
		newTextString += (dataRows == 0) ? String.format("%02d", Integer.valueOf(0)) : (dataRows < 3) ? String.format("%02d", Integer.valueOf(3)) : (dataRows > 90) 
				? String.format("%02d", Integer.valueOf(90)) : String.format("%02d", dataRows);
		//Add data col
		newTextString += ((dataCols == 0) ? String.format("%02d", Integer.valueOf(0)) : (dataCols > 30) ? String.format("%02d", Integer.valueOf(30)) 
				: String.format("%02d", dataCols))+ textString;

		writeBarCode("z", newTextString, row, col, parameters);
	}
	/** 
	 This function prints a DataMatrix Bar Code, a 2-D matrix symbology comprised of square mdoules arranged within a perimeter finder pattern.
	 <p>Two basic types:</p>
	 <p>ECC 000-140</p>
	 <p>ECC 200</p>
	 
	 @param textString Any 8-byte data
	 @param correctionLevel 3-Digit convolutional error correction level(ECC).Valid values: 0,50,80,100,140,200
	 @param digitFormatID 1-digit format Identification:
	 <p> 0 - Automatically choose encodation scheme base on characters to be encoded</p>
	 <p> 1 - Numeric</p>
	 <p> 2 - Uppercase alphabetic</p>
	 <p> 3 - Uppercase alphanumeric and punctuation characters(.,-/)</p>
	 <p> 4 - Uppercase alphanumeric</p>
	 <p> 5 - ASCII,the full 128 ASCII char set</p>
	 <p> 6 - Any 8-bit byte</p>
	 <p>For ECC 200, value should be 0</p>
	 @param rowsRequested Odd or even number of rows requested. Valid Values:
	 <p> 0 - Automatically determined</p>
	 <p> For ECC 000-140: 9, 11, 13,..49</p>
	 <p> For ECC 200: 10,12,14,16,1,20,22,24,26,32,36,40,44,48,52,64,72,80,96,104,120,132,144 </p>
	 
	 @param colsRequested Odd or even number of cols requested. Valid Values:
	 <p> 0 - Automatically determined</p>
	 <p> For ECC 000-140: 0 , 9, 11, 13,..49</p>
	 <p> For ECC 200: 10,12,14,16,1,20,22,24,26,32,36,40,44,48,52,64,72,80,96,104,120,132,144 </p>
	 
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeBarCodeDataMatrix(String textString, int correctionLevel, int digitFormatID, int rowsRequested, int colsRequested, int row, int col, ParametersDPL parameters)
	{
		parameters.setSymbolHeight(0);
		//Validate correctionLevel
		if (correctionLevel < 0 || correctionLevel > 200)
		{
			throw new IllegalArgumentException("Parameter 'correctionLevel' must be between 0-200. Value given was " + Integer.valueOf(correctionLevel).toString());
		}
		//Validate digitFOrmatID
		if (digitFormatID < 0 || digitFormatID > 6)
		{
			throw new IllegalArgumentException("Parameter 'digitFormatID' must be between 0-6. Value given was " + Integer.valueOf(digitFormatID).toString());
		}

		//Validate rowsRequested
		if (rowsRequested !=0 && (rowsRequested < 9 || rowsRequested > 144))
		{
			throw new IllegalArgumentException("Parameter 'rowsRequested' must be 0 or between 9-144. Value given was " + Integer.valueOf(rowsRequested).toString());
		}
		//Validate colsRequested
		if (colsRequested != 0 &&(colsRequested <9 || colsRequested > 144))
		{
			throw new IllegalArgumentException("Parameter 'colsRequested' must be 0 or between 9-144. Value given was " + Integer.valueOf(colsRequested).toString());
		}
		writeBarCode("W1c", String.format("%03d",correctionLevel) + ((correctionLevel == 200)?"0":Integer.valueOf(digitFormatID).toString()) 
				+ String.format("%03d",rowsRequested) + String.format("%03d",colsRequested) + textString, row, col, parameters);

	}
	/** 
	 This prints a QRCode Bar Code.
	 
	 @param textString Numeric Data, Alphanumeric Data, 8-bit byte data, and Kanji characters
	 @param autoFormatting Specifies to use auto or manual QR bar code formatting.If this is true, other parameters will be ignored.
	 @param QRCodeModel QR Code Model number, optional. Model 2 is default
	 @param correctionLevel Error Correction Levels:
	 <p>"H" = Ultra Reliability Level (30%)</p>
	 <p>"Q" = High Reliability Level (25%)</p>
	 <p>"M" = Standard Reliability Level (15%)</p>
	 <p>"L" = High Density Level (7%)</p>
	 @param maskNumber Mask Number,optional:
	 <p>None = Auto</p>
	 <p>0-7 = Mask 0 to Mask 7</p>
	 <p>8 = No Mask</p>
	 @param inputMode Data Input Mode:
	 <p>A = Auto, ASCII</p>
	 <p>a = Auto,hex-ASCII</p>
	 <p>M = Manual, ASCII</p>
	 <p>m = Manual, hex-ASCII</p>
	 
	 @param charMode Character Mode: 
	 <p>N = Numeric</p>
	 <p>A = Alphanumeric</p>
	 <p>B = Binary (where nnnn = data bytecount,4 decimal digits; byte-count /2 for hex-ASCII)</p>
	 <p>K = Kanji</p>
	 
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeBarCodeQRCode(String textString, boolean autoFormatting, int QRCodeModel, String correctionLevel, String maskNumber, String inputMode, String charMode, int row, int col, ParametersDPL parameters)
	{
		parameters.setSymbolHeight(0);

		if (autoFormatting)
		{
			writeBarCode("W1d", textString + "\r", row, col, parameters);
		}
		else
		{
			//Validate QRCodeModel
			if (QRCodeModel != 1 && QRCodeModel != 2)
			{
				throw new IllegalArgumentException("Parameter 'QRCodeModel' must be 1 or 2.Value given was " + Integer.valueOf(QRCodeModel).toString());
			}
			//Validate CorrectionLevel
			if (!correctionLevel.equals("H") && !correctionLevel.equals("Q") && !correctionLevel.equals("M") && !correctionLevel.equals("L"))
			{
				throw new IllegalArgumentException("Parameter 'correctionLevel' must be H,Q,M, or L. Value given was " + correctionLevel);
			}
			//Validate MaskNumber
			if (!Document.matches(maskNumber,"^[0-8]{1}$") && !maskNumber.equals(""))
			{
				throw new IllegalArgumentException("Parameter 'Mask Number' must be between 0-8 or blank. Value given was " + maskNumber.toString());
			}
			//Validate Input MOde
			if (!inputMode.equals("A") && !inputMode.equals("a") && !inputMode.equals("M") && !inputMode.equals("m"))
			{
				throw new IllegalArgumentException("Parameter 'inputMode' must be A,a,M, or m. Value given was " + inputMode);
			}
			//Validate Character Mode
			if (!charMode.equals("N") && !charMode.equals("A") && !charMode.equals("B") && !charMode.equals("K"))
			{
				throw new IllegalArgumentException("Parameter 'charMode' must be N,A,B, or K. Value given was " + charMode);
			}

			writeBarCode("W1D", Integer.valueOf(QRCodeModel).toString() + ',' + correctionLevel + maskNumber.toString() + inputMode + ',' + charMode + textString, row, col, parameters);
		}
	}
	/** 
	 This prints an Aztec Bar code.
	 
	 @param textString ALl ASCII characters, depending on selected options
	 @param textLength Optional string length specifier. Set to 0 if not used
	 @param ECIMode Extended Channel Interpretation mode; 0 = Disabled, 1 = Enabled
	 @param errorCorrection Error Correction(EC)/Amount:
	 <p>0:Default</p>
	 <p>1-99: EC fiexed value,expressed as percent</p>
	 <p>101-104: Compact Core, 1 to 4 layers </p>
	 <p>201-232: Full size core,1 to 32 layers</p>
	 <p>300:Rune format, encodes 3 ASCII 0-256 decimal digits.</p>
	 
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeBarCodeAztec(String textString, int textLength, boolean ECIMode, int errorCorrection, int row, int col, ParametersDPL parameters)
	{
		parameters.setSymbolHeight(0);


		//Validate textLength
		if (textLength < 0 || textLength > 9999)
		{
			throw new IllegalArgumentException("Parameter 'textLength' must be between 0-9999. Value given was " + Integer.valueOf(textLength).toString());
		}
		//Validate errorCorrection
		if (errorCorrection < 0 || errorCorrection > 300)
		{
			throw new IllegalArgumentException("Parameter 'errorCorrection' must be between 0-300. Value given was " + Integer.valueOf(errorCorrection).toString());
		}


		writeBarCode("W1f", String.format("%04d", textLength) + (ECIMode?Integer.valueOf(1).toString():Integer.valueOf(0).toString())
				+ String.format("%03d",errorCorrection) + textString, row, col, parameters);

	}
	/** 
	 Prints a GS1 DataBar Bar Code.
	 <p>GS1 DataBar Truncated/Stacked/Omni-Directional: 14-Digit EAN.UCC item ID number with linear symbol</p>
	 <p>   Encodable character set 0 - 9</p>
	 <p>   Maximum numeric data capacity is the application identifier plus 14-digit</p>
	 <p>   Error detection is mod 79 checksum</p>
	 <p>GS1 DataBar Limited - 14-Digit EAN.UCC item ID number with 0 or 1 within linear symbol</p>
	 <p>   Encodable character set 0 - 9</p>
	 <p>   Maximum numeric data capacity is the application identifier plus 14-digit</p>
	 <p>   Data must begin with 0 or 1</p>
	 <p>   Error detection is mod 89 checksum</p>
	 <p>GS1 DataBar Expanded - 14-Digit EAN.UCC item ID number plus supp. AI element strings</p>
	 <p>   Encodable character set ISO 646(alphanumeric,20 punctuation characters, and FNC1)</p>
	 <p>   Maximum numeric data capacity is 74 numeric or 41 alphanumeric</p>
	 <p>   Error detection is mod 211 checksum</p>
	 
	 @param textString Additional 2-D data
	 @param barCodeData Numeric linear data(length 13) or ISO646 character set(GS1 Expanded)
	 @param GS1Type GS1 Type:
	 <p>E = GS1 DataBar Expanded</p>
	 <p>R = GS1 DataBar OmniDirectional</p>
	 <p>T = GS1 DataBar Truncated</p>
	 <p>S = GS1 DataBar Stacked</p>
	 <p>D = GS1 DataBar Stacked OmniDirectional</p>
	 <p>L = GS1 DataBar Limited</p>
	 
	 @param pixelMultiplier Pixel Multiplier (1-9)
	 @param xPixelUndercut X Pixels to undercut ( 0 to (pixelmultiplier - 1))
	 @param yPixelUndercut Y Pixels to undercut ( 0 to (pixelmultiplier - 1))
	 @param segmentsPerRow Segments per row for GS1 DataBar Expanded. Valid values 2-22, even. For other types, use 0.
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeGS1DataBar(String textString, String barCodeData, String GS1Type, int pixelMultiplier, int xPixelUndercut, int yPixelUndercut, int segmentsPerRow, int row, int col, ParametersDPL parameters)
	{
		parameters.setSymbolHeight(0);

		//Validate GS1 Type
		if (!GS1Type.equals("E") && !GS1Type.equals("R") && !GS1Type.equals("T") && !GS1Type.equals("S") && !GS1Type.equals("D") && !GS1Type.equals("L"))
		{
			throw new IllegalArgumentException("Parameter 'GS1Type' must be E,R,T,S,D, or L. Value given was " + GS1Type);
		}

		//Validate Pixel Multiplier
		if (pixelMultiplier < 1 || pixelMultiplier > 9)
		{
			throw new IllegalArgumentException("Parameter 'pixelMultiplier' must be between 1-9. Value given was " + Integer.valueOf(pixelMultiplier).toString());
		}

		//Validate x pixel undercut
		if (xPixelUndercut < 0 || xPixelUndercut > (pixelMultiplier - 1))
		{
			throw new IllegalArgumentException("Parameter 'xPixelUndercut' must be between 0 and pixelMultiplier value. Value given was " + Integer.valueOf(xPixelUndercut).toString());
		}

		//Validate y pixel undercut
		if (yPixelUndercut < 0 || yPixelUndercut > (pixelMultiplier - 1))
		{
			throw new IllegalArgumentException("Parameter 'yPixelUndercut' must be between 0 and pixelMultiplier value. Value given was " + Integer.valueOf(yPixelUndercut).toString());
		}

		//Validate barCodeData data
		//If barcode type is GS1 Expanded
		if (GS1Type.equals("E"))
		{
			if (!Document.matches(barCodeData, "^[a-zA-Z0-9!%&\\?\'\"\\-\\./,:;<=>_ ]*$"))
			{
				throw new IllegalArgumentException("Parameter 'barcodeData' must contain only alphanumeric characters or the following special characters: !%&?'\"-.,;:<=>_ . Value given was " + barCodeData);
			}
		}
		//For all other GS1 DataBar Codes
		else
		{
			if (!Document.matches(barCodeData, "^[0-9]{13}$"))
			{
				throw new IllegalArgumentException("Parameter 'barcodeData' must be numeric and 13 digits. Value given was " + barCodeData);
			}
		}

		//Validate segment per row
		if ((segmentsPerRow % 2) != 0 || (segmentsPerRow < 0 || segmentsPerRow > 22))
		{
			throw new IllegalArgumentException("Parameter 'segmentsPerRow' must be even and no greater than 22. Value given was " + Integer.valueOf(segmentsPerRow).toString());
		}

		//form GS1 barcode content
		String newTextString = (textString.equals(""))? (GS1Type + Integer.valueOf(pixelMultiplier).toString() 
				+ Integer.valueOf(xPixelUndercut).toString() + Integer.valueOf(yPixelUndercut).toString() 
				+ ((segmentsPerRow != 0)? String.format("%02d",segmentsPerRow):"") + barCodeData) 
				: (GS1Type + Integer.valueOf(pixelMultiplier).toString() + Integer.valueOf(xPixelUndercut).toString()+ Integer.valueOf(yPixelUndercut).toString() 
				+ ((segmentsPerRow != 0)?String.format("%02d", segmentsPerRow):"") + barCodeData + '|' + textString);

		writeBarCode("W1k", newTextString, row, col, parameters);
	}
	/** 
	 This prints an Austrailia Post 4-State Bar Code
	 
	 @param textString Optional. Enter customer information here
	 @param readableText Specifies whether to print human readable field or not
	 @param formatControlCode FOrmat COntrol Code. Valid values:
	 <p>11 = Standard Customer Bar Code</p>
	 <p>87 = Routing Bar Code</p>
	 <p>45 = Reply Paid Bar Code</p>
	 <p>92 = Redirection Bar Code</p>
	 <p>59 = Customer Bar Code 2</p>
	 <p>62 = Customer Bar Code 3</p>
	 <p>44 = Reserved</p>
	 
	 @param DPID 8-digit Delivery Point Identifier
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeBarCodeAusPost4State(String textString, boolean readableText, int formatControlCode, int DPID, int row, int col, ParametersDPL parameters)
	{
		parameters.setNarrowBarWidth(0);
		parameters.setWideBarWidth(0);
		parameters.setSymbolHeight(0);
		//Validate formatControlCode
		if (formatControlCode != 11 && formatControlCode != 44 && formatControlCode != 45 && formatControlCode != 59 && formatControlCode != 62 && formatControlCode != 87 && formatControlCode != 92)
		{
			throw new IllegalArgumentException("Parameter 'formatControlCode' must be 11,44,45,59,62,87,92. Value given was " + Integer.valueOf(formatControlCode).toString());
		}
		//Validate DPID value
		if (DPID < 0 || DPID > 99999999)
		{
			throw new IllegalArgumentException("Parameter 'DPID' must be between 0-99999999. Value given was " + Integer.valueOf(DPID).toString());
		}
		//for customer barcode 2
		if (formatControlCode == 59)
		{
			//Checks if it contains alpha numeric characters
			if (Document.matches(textString, "^[0-9a-zA-Z]*$"))
			{
				//if it only contains numbers
				if (Document.matches(textString, "^[0-9]*$"))
				{
					if (textString.length() > 8)
					{
						throw new IllegalArgumentException("Parameter 'textString' contains numeric digit.Must be a maximum 8 digits for BarCode 2.");
					}
				}
				else
				{
					if (textString.length() > 5)
					{
						throw new IllegalArgumentException("Parameter 'textString' contains alphanumeric characters.Must be a maximum 5 alphanumeric characters for BarCode 2.");
					}
				}
			}
			//invalid input
			else
			{
				throw new IllegalArgumentException("Parameter 'textString' must contains a maximum of 5 alphanumeric characters or 8 numeric digits for BarCode 2.");
			}

		}
		//Customer barcode 3
		else if (formatControlCode == 62)
		{
			//Checks if it contains alpha numeric characters
			if (Document.matches(textString, "^[0-9a-zA-Z]*$"))
			{
				//if it only contains numbers
				if (Document.matches(textString, "^[0-9]*$"))
				{
					if (textString.length() > 15)
					{
						throw new IllegalArgumentException("Parameter 'textString' contains numeric digit.Must be a maximum 15 digits for BarCode 3.");
					}
				}
				else
				{
					if (textString.length() > 10)
					{
						throw new IllegalArgumentException("Parameter 'textString' contains alphanumeric characters.Must be a maximum 10 alphanumeric characters for BarCode 3.");
					}
				}
			}
			//invalid input
			else
			{
				throw new IllegalArgumentException("Parameter 'textString' must contains a maximum of 10 alphanumeric characters or 15 numeric digits for BarCode 3.");
			}

		}

		//write barcode
		writeBarCode((readableText) ? "W1M" : "W1m", Integer.valueOf(formatControlCode).toString()+ String.format("%08d", DPID) + textString, row, col, parameters);
	}
	/** 
	 This prints a CODABLOCK Bar code
	 
	 @param textString CODABLOCK A: 0-9, A-Z, - . * $ / + % and the space character; and CODABLOCK E and F: All ASCII characters.
	 @param rowHeight Individual Row Height
	 @param codaBlockMode Codeblock mode: A,E, or F
	 @param generateCheckSum Specifies whether to genrate and add checksum
	 @param rowsToEncode Number of rows to be encoded
	 @param charPerRow Number of characters per row
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeBarCodeCODABLOCK(String textString, int rowHeight, String codaBlockMode, boolean generateCheckSum, int rowsToEncode, int charPerRow, int row, int col, ParametersDPL parameters)
	{
		parameters.setSymbolHeight(rowHeight);
		//Validate mode
		if (!codaBlockMode.equals("A") && !codaBlockMode.equals("E") && !codaBlockMode.equals("F"))
		{
			throw new IllegalArgumentException("Parameter 'codaBlockMode' must be A,E, or F. Value given was " + codaBlockMode);
		}
		//Validate number of rows to encode
		if (rowsToEncode < 1 || rowsToEncode> 44)
		{
			throw new IllegalArgumentException("Parameter 'rowsToEncode' must be between 1-44. Value given was " + Integer.valueOf(rowsToEncode).toString());
		}

		//Validate number of characters per row
		if (charPerRow < 2 || charPerRow > 62)
		{
			throw new IllegalArgumentException("Parameter 'charPerRow' must be between 2-62. Value given was " + Integer.valueOf(charPerRow).toString());
		}


		writeBarCode("W1q", codaBlockMode + (generateCheckSum?Integer.valueOf(1).toString():Integer.valueOf(0).toString())
				+ String.format("%02d", rowsToEncode) + String.format("%02d",charPerRow)
				+ textString, row, col, parameters);

	}
	/** 
	 This prints a TCIF Linked 3 of 9 Barcode
	 
	 @param serialNumber Up to 25 alphanumeric serial character tobe encoded in the MicroPDF417 symbol
	 @param barCodeHeight Height of COde 39
	 @param ECINumber Six Digit ECI Number. If input is less than 6 dgits(e.g 1,10,200, etc..), then it will
	 automatically converted to 6-digit format(eg. 000200)
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeBarCodeTLC39(String serialNumber, int barCodeHeight, int ECINumber, int row, int col, ParametersDPL parameters)
	{
		parameters.setSymbolHeight(barCodeHeight);

		//Validate ECINumber
		if (ECINumber < 0 || ECINumber > 999999)
		{
			throw new IllegalArgumentException("Parameter 'ECINumber' must be between 0-999999. Value given was " + ECINumber);
		}
		//Validate serialNumber
		if (!Document.matches(serialNumber,"^[a-zA-Z0-9]*$") || (serialNumber.length() > 25))
		{
			throw new IllegalArgumentException("Parameter 'serialNumber' must be 0-25 alphanumeric characters. Value given was " + serialNumber);
		}

		writeBarCode("W1t", Integer.valueOf(ECINumber).toString() + ';' + serialNumber, row, col, parameters);
	}
	/** 
	 This prints a MicroPDF417 2-D bar code holding large amounts of data in small area
	 
	 @param textString Data to be encoded
	 @param dataCols Number Columns. Valid values 1-4
	 @param rowIndex Row/Error Correction Index. Valid values 1-10
	 @param byteCompactMode Specifies to use Byte Compaction Mode, true = best binary data compression
	 @param macroCharSub Specifies to use Macro Character Substitution, true = disable
	 @param row Row position, starting from zero, to start 
	 printing at.
	 @param col Column position, starting from zero, to start 
	 printing at.
	 @param parameters Parameters DPL object to specify other printing properties
	*/
	public void writeBarCodeMicroPDF417(String textString, int dataCols, int rowIndex, boolean byteCompactMode, boolean macroCharSub, int row, int col, ParametersDPL parameters)
	{
		parameters.setSymbolHeight(0);

		//Validate rowIndex
		if (rowIndex < 0 || rowIndex > 10)
		{
			throw new IllegalArgumentException("Parameter 'rowIndex' must be between 0-10. Value given was " + Integer.valueOf(rowIndex).toString());
		}

		//Validate data Cols
		if (dataCols < 1 || dataCols > 4)
		{
			throw new IllegalArgumentException("Parameter 'dataCols' must be between 1-4. Value given was " + Integer.valueOf(dataCols).toString());
		}

		//Start textstring
		String newTextString = Integer.valueOf(dataCols).toString() + ((rowIndex == 10) ? "A" : Integer.valueOf(rowIndex).toString())
				+ (byteCompactMode?Integer.valueOf(1).toString():Integer.valueOf(0).toString()) 
				+ (macroCharSub?Integer.valueOf(1).toString():Integer.valueOf(0).toString())
				+ '0' + textString;

		writeBarCode("W1z", newTextString, row, col, parameters);
	}
	/** 
	 This function will print the provided text to the document as a
	 barcode using the font specified and the provided printing parameter
	 values.
	 
	 @param barcodeID ID of the barcode font. Valid values: A to Z(bc with human readable text), a to z (non-human readable text)(except P,u,v,z)
	 or Wna(n = 1-9, a = a-s/A-S, No n is an implied 1)
	 @param textString Text to print as barcode.
	 @param row Row position, starting from zero, to start printing
	 at.
	 @param col Column position, starting from zero, to start
	 printing at.
	 @param parameters This ParametersDPL object specifies any 
	 printing parameter values you wish to alter for the printing 
	 of this item.
	*/
	public void writeBarCode(String barcodeID, String textString, int row, int col, ParametersDPL parameters)
	{

		// Validate position (throws on error)
		ValidatePosition(row, col);


		// Check inputs
		if ((barcodeID == null) || ((barcodeID.length() != 0) && (barcodeID.length() > 3)))
		{
			// Out of range
				throw new IllegalArgumentException(String.format("Parameter 'barcodeID' must be at least %1$s characters in length.", 3));
		}
		if (textString == null)
		{
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'textString' was null.");
		}

		if (parameters == null)
		{
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}
		// Add to Document
		byte[] tempArray;

		//Get parameter string
		tempArray = parameterString(barcodeID, parameters, PrintingType.Barcode);
		
		addToDoc(m_Document, tempArray);
		//write row and col postion
		addToDoc(m_Document,String.format("%04d", row) + String.format("%04d", col));
		//write Text string
		addToDoc(m_Document, textString);
		addToDoc(m_Document, EOL);

		return;
	}
	/** 
	 This will cause the image stored on the printer to be printed 
	 at the given location.
	 
	 @param imageName Name of image stored on printer.
	 @param row Row position, starting from zero, to start printing at.
	 @param col Column position, starting from zero, to start printing at.
	 @param parameters This ParametersDPL object specifies any
	 printing parameter values you wish to alter for the printing
	 of this item.
	*/
	public void writeImageStored(String imageName, int row, int col, ParametersDPL parameters)
	{
		parameters.setRotate(ParametersDPL.Rotation.Rotate_0);
		parameters.setFont("000");
		byte[] tempArray;
		//Start label <STX>L<CR>
		addToDoc(m_Document,String.format("\u0002" + "L\r\n"));
		//Set Label format
		addToDoc(m_Document,"A" + Integer.valueOf(m_LabelFormat.getValue()).toString() + "\r\n");

		//===========Set global parameters settings===================/
		//Set Column offset
		if (getColumnOffset() > 0)
		{
			addToDoc(m_Document,"C" + String.format("%04d", getColumnOffset()) + "\r\n");
			
		}
		//Set RowOffset
		if (getRowOffset() > 0)
		{
			addToDoc(m_Document,"R" + String.format("%04d", getRowOffset()) + "\r\n");
			
		}

		//Set DotHeightMultiplier
		addToDoc(m_Document,"D" + Integer.valueOf(getDotWidthMultiplier()).toString() + Integer.valueOf(getDotHeightMultiplier()).toString() + "\r\n");
		


		//Set Heat Settings
		if (getHeatSettings() != 10)
		{
			addToDoc(m_Document,"H" + String.format("%02d", getHeatSettings()) + "\r\n");
			
		}


		//Get parameter string
		tempArray = parameterString("Y", parameters, PrintingType.Image);
		
		addToDoc(m_Document, tempArray);
		//write row and col position
		addToDoc(m_Document,String.format("%04d", row) + String.format("%04d", col));
		//write the following image name to printer
		addToDoc(m_Document, imageName);
		addToDoc(m_Document, EOL);
		// End of job
		addToDoc(m_Document,"E\r\n");
		

	}

	/** 
	 This will cause the image file specified to be printed at the given location.
	 
	 @param imagePath File name or path of image to be printed.
	 @param imageType Format of the image to be printed.
	 @param row Row position, starting from zero, to start printing at.
	 @param col Column position, starting from zero, to start printing at.
	 @param parameters This ParametersDPL object specifies any
	 printing parameter values you wish to alter for the printing
	 of this item.
	 * @throws IOException 
	 * @throws Exception 
	*/
	public void writeImage(String imagePath, ImageType imageType, int row, int col, ParametersDPL parameters) throws Exception
	{	
		//Create bitmap and bitmap data for original image file
		BufferedImage imageBitmap =ImageIO.read(new File(imagePath));
		byte[] bitmapData = {0};

		//If image cannot be converted into bitmap, try to read file as is and write to printer
		if (imageBitmap == null)
		{
			//Read all bytes of the file
			
			File imageFile = new File(imagePath);
   		bitmapData = new byte[(int)imageFile.length()];
   		InputStream inputStream= new BufferedInputStream(new FileInputStream(imageFile));
   		inputStream.read(bitmapData);
   		inputStream.close();

			writeImage(bitmapData, imageType, row, col, parameters);
		}
		//Write bitmap to printer
		else
		{
			writeImage(imageBitmap, imageType, row, col, parameters);
		}

		return;
	}
	
	
	/** 
	 This will cause the bitmap specified to be printed at the given location.
	 
	 @param imageObject Image data to be printed.
	 @param imageType Format of the image to be printed.
	 @param row Row position, starting from zero, to start printing
	 at.
	 @param col Column position, starting from zero, to start
	 printing at.
	 @param parameters This ParametersDPL object specifies any
	 printing parameter values you wish to alter for the printing
	 of this item.
	*/
	public void writeImage(BufferedImage imageObject, ImageType imageType, int row, int col, ParametersDPL parameters) throws Exception
	{
		if (imageObject == null)
		{
			//Invalid image object
			throw new IllegalArgumentException("Parameter 'imageObject' was null.");
		}
		if (parameters == null)
		{
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}
		 //==================Convert image to monochrome bitmap 1bpp if not in recognized format=======================================//
       //Change image type
       imageType = ImageType.BMPFlipped_8Bit;
       
       MonoBitmapConverter monoBMconv = new MonoBitmapConverter();
       
       byte[] bitmapData = monoBMconv.convertBitmap(imageObject);

       //===========================================================================================================================//
     //Call out full function
       writeImage(bitmapData, imageType, row, col, parameters);
		return;
	}
	/** 
	 This will cause the image data specified to be printed at the given location.
	 
	 @param imageData Image data to be printed. For printing images stored on printer, set value to null
	 @param imageType Format of the image to be printed.
	 @param row Row position, starting from zero, to start printing
	 at.
	 @param col Column position, starting from zero, to start
	 printing at.
	 @param parameters This ParametersDPL object specifies any
	 printing parameter values you wish to alter for the printing
	 of this item.
	*/
	public void writeImage(byte[] imageData, ImageType imageType, int row, int col, ParametersDPL parameters) throws Exception
	{
		// Validate position (throws on error)
		ValidatePosition(row, col);
		if (imageData == null)
		{
			//Invalid image object
			throw new IllegalArgumentException("Parameter 'imageData' was null.");
		}
		if (parameters == null)
		{
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}
		
		parameters.setRotate(Rotation.Rotate_0);
		parameters.setFont("000");
		// Add to Document
		byte[] tempArray;
		//Clear Module E
		addToDoc(m_Document,String.format("\u0002" + "qE\r\n"));
		
		//Set the default module to E
		addToDoc(m_Document,String.format("\u0002" + "XE\r\n"));
		

		//write Image Input command to to printer
		addToDoc(m_Document,getInputImageDataCommand("image_downloaded", imageType));
		

		//write image data followed by CR to printer
		addToDoc(m_Document,imageData);
		addToDoc(m_Document, EOL);


		//Start label <STX>L<CR>
		addToDoc(m_Document,String.format("\u0002" + "L\r\n"));
		//Set Label format
		addToDoc(m_Document,"A" + Integer.valueOf(m_LabelFormat.getValue()).toString() + "\r\n");

		//===========Set global parameters settings===================/
		//Set Column offset
		if (getColumnOffset() > 0)
		{
			addToDoc(m_Document,"C" + String.format("%04d", getColumnOffset()) + "\r\n");
		}
		//Set RowOffset
		if (getRowOffset() > 0)
		{
			addToDoc(m_Document,"R" + String.format("%04d", getRowOffset()) + "\r\n");
		}

		//Set DotHeightMultiplier
		addToDoc(m_Document,"D" + Integer.valueOf(getDotWidthMultiplier()).toString() + Integer.valueOf(getDotHeightMultiplier()).toString() + "\r\n");

		//Set Heat Settings
		if (getHeatSettings() != 10)
		{
			addToDoc(m_Document,"H" + String.format("%02d", getHeatSettings()) + "\r\n");
		}

		//Get parameter string
		tempArray = parameterString("Y", parameters, PrintingType.Image);
		
		addToDoc(m_Document, tempArray);
		//write row and col position
		addToDoc(m_Document,String.format("%04d", row) + String.format("%04d", col));
		//write the following image name to printer
		addToDoc(m_Document, "image_downloaded");
		addToDoc(m_Document, EOL);
		// End of job
		addToDoc(m_Document,"E\r\n");
		
		//Clear Module E
		addToDoc(m_Document,String.format("\u0002" + "qE\r\n"));
		//===============================================================================================//
		return;
	}
	
//	protected void SetIndexedPixel(int x, int y,Bitmap boolean setPixel)
//	{
//		int index = y * bm.Stride + (x >> 3);
//		byte p = Marshal.ReadByte(bmd.Scan0, index);
//		byte mask = (byte)(0x80 >> (x & 0x7));
//
//		if (setPixel)
//		{
//			p |= mask;
//		}
//		else
//		{
//			p &= (byte)(mask ^ 0xff);
//		}
//		Marshal.writeByte(bmd.Scan0, index, p);
//	}

	/** 
	 This method will cause the Line specified to be printed.
	 
	 @param row Row position, starting from zero, to start printing
	 at.
	 @param col Column position, starting from zero, to start
	 printing at.
	 @param height height of the line
	 @param width Width of the line
	*/
	public void writeLine(int row, int col, int height, int width)
	{
		ParametersDPL parameters = new ParametersDPL();
		parameters.setFillPattern(0);
		parameters.setRotate (ParametersDPL.Rotation.Rotate_0);
		parameters.setHorizontalMultiplier(1);
		parameters.setVerticalMultiplier(1);

		//Validate height and width
		if (width < 0 || width > 9999)
		{
			throw new IllegalArgumentException("Parameter 'width' must be between 0 and 9999.");
		}
		if (height < 0 || height > 9999)
		{
			throw new IllegalArgumentException("Parameter 'height' must be between 0 and 9999.");
		}
		String lineData = "l" + String.format("%04d", width) + String.format("%04d", height);
		writeGraphics(lineData, row, col, parameters);
	}
	/** 
	 This method will cause the box specified to be printed.
	 
	 @param row - Row position, starting from zero, to start printing at
	 @param col - Col position, starting from zero, to start printing at
	 @param height - Height of the box
	 @param width - Width of the box
	 @param topBottomThickness - The thickness of the top and bottom edges
	 @param sideThickness - The thickness of the sides
	*/
	public void writeBox(int row, int col, int height, int width, int topBottomThickness, int sideThickness)
	{
		ParametersDPL parameters = new ParametersDPL();
		parameters.setFillPattern(0);
		parameters.setRotate (ParametersDPL.Rotation.Rotate_0);
		parameters.setHorizontalMultiplier(1);
		parameters.setVerticalMultiplier(1);

		//Validate height and width
		if (width < 0 || width > 9999)
		{
			throw new IllegalArgumentException("Parameter 'width' must be between 0 and 9999.");
		}
		if (height < 0 || height > 9999)
		{
			throw new IllegalArgumentException("Parameter 'height' must be between 0 and 9999.");
		}
		if (topBottomThickness < 0 || topBottomThickness > 9999)
		{
			throw new IllegalArgumentException("Parameter 'topBottomThickness' must be between 0 and 9999.");
		}
		if (sideThickness < 0 || sideThickness > 9999)
		{
			throw new IllegalArgumentException("Parameter 'sideThickness' must be between 0 and 9999.");
		}
		String boxData = "b" + String.format("%04d", width) +String.format("%04d", height) + String.format("%04d", topBottomThickness) + String.format("%04d", sideThickness);

		writeGraphics(boxData, row, col, parameters);

	}
	/** 
	 This will cause the specified triangle to be printed
	 
	 @param fillPattern Fill pattern of the specified triangle
	 <p>Valid values:</p>
	 <p> 0 = No Patterns</p>
	 <p> 1 = Solid Black</p>
	 <p> 2 = 6% Black</p>
	 <p> 3 = 12% Black</p>
	 <p> 4 = 25% Black</p>
	 <p> 5 = 38% Black</p>
	 <p> 6 = 50% Black</p>
	 <p> 7 = Diamonds</p>
	 <p> 8 = Circles</p>
	 <p> 9 = Right Diagonal Lines</p>
	 <p> 10 = Left Diagonal Lines</p>
	 <p> 11 = Grid</p>
	 
	 @param rowPt1 Row position of the first point of triangle
	 @param colPt1 Column position of the first point of the triangle
	 @param rowPt2 Row position of the second point of triangle
	 @param colPt2 Column position of the second point of the triangle
	 @param rowPt3 Row position of the third point of triangle
	 @param colPt3 Column position of the third point of the triangle
	*/
	public void writeTriangle(int fillPattern, int rowPt1, int colPt1, int rowPt2, int colPt2, int rowPt3, int colPt3)
	{
		ParametersDPL parameters = new ParametersDPL();
		parameters.setFillPattern(fillPattern);
		parameters.setRotate (ParametersDPL.Rotation.Rotate_0);
		parameters.setHorizontalMultiplier(1);
		parameters.setVerticalMultiplier(1);

		//Validate height and width
		if (rowPt2 < 0 || rowPt2 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'rowPt2' must be between 0 and 9999.");
		}
		if (colPt2 < 0 || colPt2 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'colPt2' must be between 0 and 9999.");
		}
		if (rowPt3 < 0 || rowPt3 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'rowPt3' must be between 0 and 9999.");
		}
		if (colPt3 < 0 || colPt3 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'colPt3' must be between 0 and 9999.");
		}
		String TriangleData = "P" + "001" + "0001" + String.format("%04d", rowPt2) + String.format("%04d", colPt2)
				+ String.format("%04d", rowPt3) + String.format("%04d", colPt3);

		writeGraphics(TriangleData, rowPt1, colPt1, parameters);

	}
	/** 
	 This will cause the specified rectangle to be printed.
	 
	 @param fillPattern Fill pattern of the specified rectangle
	 <p>Valid values:</p>
	 <p> 0 = No Patterns</p>
	 <p> 1 = Solid Black</p>
	 <p> 2 = 6% Black</p>
	 <p> 3 = 12% Black</p>
	 <p> 4 = 25% Black</p>
	 <p> 5 = 38% Black</p>
	 <p> 6 = 50% Black</p>
	 <p> 7 = Diamonds</p>
	 <p> 8 = Circles</p>
	 <p> 9 = Right Diagonal Lines</p>
	 <p> 10 = Left Diagonal Lines</p>
	 <p> 11 = Grid</p>
	 
	 @param rowPt1 Row position of the first point of triangle
	 @param colPt1 Column position of the first point of the triangle
	 @param rowPt2 Row position of the second point of triangle
	 @param colPt2 Column position of the second point of the triangle
	 @param rowPt3 Row position of the third point of triangle
	 @param colPt3 Column position of the third point of the triangle
	 @param rowPt4 Row position of the fourth point of triangle
	 @param colPt4 Column position of the fourth point of the triangle
	*/
	public void writeRectangle(int fillPattern, int rowPt1, int colPt1, int rowPt2, int colPt2, int rowPt3, int colPt3, int rowPt4, int colPt4)
	{
		ParametersDPL parameters = new ParametersDPL();
		parameters.setFillPattern(fillPattern);
		parameters.setRotate (ParametersDPL.Rotation.Rotate_0);
		parameters.setHorizontalMultiplier(1);
		parameters.setVerticalMultiplier(1);

		//Validate height and width
		if (rowPt2 < 0 || rowPt2 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'rowPt2' must be between 0 and 9999.");
		}
		if (colPt2 < 0 || colPt2 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'colPt2' must be between 0 and 9999.");
		}
		if (rowPt3 < 0 || rowPt3 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'rowPt3' must be between 0 and 9999.");
		}
		if (colPt3 < 0 || colPt3 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'colPt3' must be between 0 and 9999.");
		}
		if (rowPt4 < 0 || rowPt4 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'rowPt4' must be between 0 and 9999.");
		}
		if (colPt4 < 0 || colPt4 > 9999)
		{
			throw new IllegalArgumentException("Parameter 'colPt4' must be between 0 and 9999.");
		}

		String rectData = "P" + "001" + "0001" + String.format("%04d", rowPt2) + String.format("%04d", colPt2)
		+ String.format("%04d", rowPt3) + String.format("%04d", colPt3) 
		+ String.format("%04d", rowPt4) + String.format("%04d", colPt4);

		writeGraphics(rectData, rowPt1, colPt1, parameters);
	}
	/** 
	 This will cause the specified circle to be printed.
	 
	 @param fillPattern Fill pattern of the specified circle
	 <p>Valid values:</p>
	 <p> 0 = No Patterns</p>
	 <p> 1 = Solid Black</p>
	 <p> 2 = 6% Black</p>
	 <p> 3 = 12% Black</p>
	 <p> 4 = 25% Black</p>
	 <p> 5 = 38% Black</p>
	 <p> 6 = 50% Black</p>
	 <p> 7 = Diamonds</p>
	 <p> 8 = Circles</p>
	 <p> 9 = Right Diagonal Lines</p>
	 <p> 10 = Left Diagonal Lines</p>
	 <p> 11 = Grid</p>
	 
	 @param centerRow Row position of the center point of circle
	 @param centerCol Column position of the center point of circle
	 @param radius Radius of circle
	*/
	public void writeCircle(int fillPattern, int centerRow, int centerCol, int radius)
	{
		ParametersDPL parameters = new ParametersDPL();
		parameters.setFillPattern(fillPattern);
		parameters.setRotate (ParametersDPL.Rotation.Rotate_0);
		parameters.setHorizontalMultiplier(1);
		parameters.setVerticalMultiplier(1);

		//Validate radius
		if (radius < 0 || radius > 9999)
		{
			throw new IllegalArgumentException("Parameter 'radius' must be between 0 and 9999.");
		}

		String circleData = "C" + "001" + "0001" + String.format("%04d",radius);

		writeGraphics(circleData, centerRow, centerCol, parameters);
	}
	/** 
	 This will cause the printer to print the specified graphic object 
	 at the given location.
	 
	 @param graphicData graphic data to print.
	 @param row Row position, starting from zero, to start printing
	 at.
	 @param col Column position, starting from zero, to start
	 printing at.
	 @param parameters This ParametersEZ object specifies any
	 printing parameter values you wish to alter for the printing
	 of this item.
	*/
	public void writeGraphics(String graphicData, int row, int col, ParametersDPL parameters)
	{
		// Validate position (throws on error)
		ValidatePosition(row, col);
		if (parameters == null)
		{
			// Invalid parameters object
			throw new IllegalArgumentException("Parameter 'parameters' was null.");
		}
		if (graphicData == null)
		{
			throw new IllegalArgumentException("Parameter 'graphicData' was null");
		}
		// Add to Document
		byte[] tempArray;

		//Get parameter string
		tempArray = parameterString("X", parameters, PrintingType.Graphics);
		addToDoc(m_Document, tempArray);

		//write row and col postion
		addToDoc(m_Document,String.format("%04d",row) + String.format("%04d",col));
		addToDoc(m_Document, graphicData);
		addToDoc(m_Document, EOL);

		return;
	}
	/** 
	 We check the row and column values in so many functions that changes
	 to the validation would be cumbersome. So I centralized it here to
	 make it easier to maintain.
	 
	 @param row Value for the starting row position
	 @param col Value for the starting column position
	*/
	private void ValidatePosition(int row, int col)
	{

		if ((row < 0) || (row > 9999))
		{
			// Row range was invalid
			throw new IllegalArgumentException(String.format("Parameter 'row' must be an integer from 0 to 9999, a value of %1$s was given.", row));
		}

		if ((col < 0) || (col > 9999))
		{
			// Column range was invalid
			throw new IllegalArgumentException(String.format("Parameter 'col' must be an integer from 0 to 9999, a value of %1$s was given.", col));
		}
		return;
	}

	/** 
	 This will build the parameters string for the current type of object
	 being rendered.
	 
	 @param parameters Parameter object to get settings
	 from.
	 @param typeID Type of object being rendered.
	 @param printingType Type of object we are printing
	*/
	private byte[] parameterString(String typeID, ParametersDPL parameters, PrintingType printingType)
	{
		String tempArray;
		ByteArrayOutputStream mStream = new ByteArrayOutputStream(128);

		// Set Alignment
		if (parameters.getAlignment() == Alignment.Center)
				addToDoc(mStream,"JC\r\n");
		else if (parameters.getAlignment() == Alignment.Right)
				addToDoc(mStream,"JR\r\n");

		//Set Mirror Mode
		if (parameters.getIsMirrored())
		{
			addToDoc(mStream,"M\r\n");

		}

		//Set Measurement Mode
		if (parameters.getMeasurement()== ParametersDPL.MeasurementMode.Metric)
				addToDoc(mStream,"m\r\n");
		String encodingValue = "";
	  //Set the encoding
	  if (parameters.getIsUnicode())
	  {
		  //Set encoding only if we are pritning Text or Barcodes
			if (printingType == PrintingType.General || printingType == PrintingType.Barcode)
			{
				encodingValue = parameters.SymbolSetToString(parameters.getDBSymbolSet());
				addToDoc(mStream,"yU" + encodingValue + "\r\n");
			}
	  }
		else
		{
			if (printingType == PrintingType.General || printingType == PrintingType.Barcode)
			{
				encodingValue = parameters.SymbolSetToString(parameters.getSBSymbolSet());
				addToDoc(mStream,"yS" + encodingValue + "\r\n");
			}
		}

		//Set Field Rotation ( a field)
		addToDoc(mStream,Integer.toString(parameters.getRotate().getValue()));


		//Set type ID field (b field)
		addToDoc(mStream,typeID);

		//Print Object specific fields
		//For barcode
		if (printingType == PrintingType.Barcode)
		{
			//Set WideBar Width (c field)
			tempArray =  (String) ((valueRemap(parameters.getWideBarWidth()) < 10) ? Integer.toString(valueRemap(parameters.getWideBarWidth())) : ((char)valueRemap(parameters.getWideBarWidth())));
			addToDoc(mStream,tempArray);

			//Set NarrowBar Width(d field)
			tempArray = (String) ((valueRemap(parameters.getNarrowBarWidth()) < 10) ?Integer.toString(valueRemap(parameters.getNarrowBarWidth())) :  ((char)valueRemap(parameters.getNarrowBarWidth())));
			addToDoc(mStream,tempArray);

			//Set Barcode Height(SymbolHeight) (eee field)
			addToDoc(mStream,String.format("%03d", parameters.getSymbolHeight()));

		}
		//For graphics
		else if (printingType == PrintingType.Graphics)
		{
			//Set horizontal multiplier(c field)
			tempArray = (String) ((valueRemap(parameters.getHorizontalMultiplier()) < 10) ? Integer.toString(valueRemap(parameters.getHorizontalMultiplier())) : ((char)valueRemap(parameters.getHorizontalMultiplier())));
			addToDoc(mStream,tempArray);

			//Set Vertical Multiplier (d field)
			tempArray = (String) ((valueRemap(parameters.getVerticalMultiplier()) < 10) ? Integer.toString(valueRemap(parameters.getVerticalMultiplier())) : ((char)valueRemap(parameters.getVerticalMultiplier())));
			addToDoc(mStream,tempArray);

			//Set Fill Pattern (eee field)
			addToDoc(mStream,String.format("%03d",parameters.getFillPattern()));
		}
		else
		{
			//Set horizontal multiplier(c field)
			tempArray = (String) ((valueRemap(parameters.getHorizontalMultiplier()) < 10) ? Integer.toString(valueRemap(parameters.getHorizontalMultiplier())) : ((char)valueRemap(parameters.getHorizontalMultiplier())));
			addToDoc(mStream,tempArray);

			//Set Vertical Multiplier (d field)
			tempArray = (String) ((valueRemap(parameters.getVerticalMultiplier()) < 10) ? Integer.toString(valueRemap(parameters.getVerticalMultiplier())) : ((char)valueRemap(parameters.getVerticalMultiplier())));
			addToDoc(mStream,tempArray);


			// Set Font name (eee field)
			//Checks if this applies to InternalBitmapp/Image
			if (parameters.getFont().equals("000"))
			{
				tempArray = parameters.getFont();
				addToDoc(mStream,tempArray);
			}
			//applies to scalable font
			else
			{
				//checks if its doubleByte or singleByte
				if (parameters.getIsUnicode())
				{
					tempArray = "u" + parameters.getFont();
					addToDoc(mStream,tempArray);
				}
				else
				{
					tempArray = "S" + parameters.getFont();
					addToDoc(mStream,tempArray);
				}
			}
		}

		return mStream.toByteArray();
	}
	/** 
	 Converts multiplier value to corresponding 1-9, A-Z, a-z value
	 
	 @param value to convert
	 @return Integer code of the character.
	*/
	private int valueRemap(int value)
	{
		if (value > 0 && value < 10)
		{
			return value;
		}
		//converts 10-35 to corresponding A-Z char value
		else if (value >= 10 && value < 36)
		{
			return (value + 55);
		}
		//36-61 to its a-z value
		else if (value >= 36 && value < 62)
		{
		  return (value + 61);
		}
		return value;
	}

	/** 
	 This method will return the rendered document in the printer's language
	 for sending directly to the printer.
	 
	 @return Rendered document in the printer's language.
	*/
	@Override
	public byte[] getDocumentData()
	{
		ByteArrayOutputStream mStream = new ByteArrayOutputStream();
		//Start label <STX>L<CR>
		addToDoc(mStream,"\u0002" + "L\r\n");
		//Set Label format
		addToDoc(mStream,"A" + getLabelFormat().getValue() + "\r\n");

		//===========Set global parameters settings===================/
		//Set Column offset
		if (getColumnOffset() > 0)
		{
			addToDoc(mStream,"C" + String.format("%04d", getColumnOffset()) + "\r\n");
		}
		//Set RowOffset
		if (getRowOffset() > 0)
		{
			addToDoc(mStream,"R" + String.format("%04d", getRowOffset()) + "\r\n");
		}

		//Set DotHeightMultiplier
		if (getDotWidthMultiplier() != 1 && getDotHeightMultiplier() != 1)
		{
			addToDoc(mStream,"D" + Integer.toString(getDotWidthMultiplier()) + Integer.toString(getDotHeightMultiplier()) + "\r\n");
		}

		//Set Heat Settings
		if (getHeatSettings() != 10)
		{
			addToDoc(mStream,"H" + String.format("%02d", getHeatSettings()) + "\r\n");
		}

		// Add document data
		addToDoc(mStream,m_Document.toByteArray());

		// End of job
		addToDoc(mStream,"E\r\n");


		//===============================================================//
		return mStream.toByteArray();
	}

	/** 
	 This method will return the rendered image in the printer's language
	 for sending directly to the printer.
	 
	 @return The render image document
	*/
	public byte[] getDocumentImageData()
	{
		ByteArrayOutputStream mStream = new ByteArrayOutputStream();

		// Add document data
		addToDoc(mStream,m_Document.toByteArray());


		//===============================================================//
		return mStream.toByteArray();
	}
	/** 
	 Returns the input image data command with the correct parameters
	 
	 @param imageName - Name of image to be download to printer
	 @param imageType - Image Format of image to be download printer
	 @return string containing the input image data command
	*/
	public String getInputImageDataCommand(String imageName, ImageType imageType)
	{
		//=====Set printer to send input image Data command
		String inputImageDataCommand = "\u0002" + "IE";
		//for 7-bit DO image
		if (imageType == ImageType.DOImage_7Bit)
		{
			//Image Data Value Range
			inputImageDataCommand += "A";
			//Specifies that we are printing 7-bit DO image. 
			inputImageDataCommand += "F";
		}
		else
		{
			switch (imageType)
			{
				case BMPFlipped_8Bit:
					//Specifies that we are printing image. 
					inputImageDataCommand += "B";
					break;
				case BMP_8Bit:
					//Specifies that we are printing image. 
					inputImageDataCommand += "b";
					break;
				case IMGFlipped_8Bit:
					//Specifies that we are printing image. 
					inputImageDataCommand += "I";
					break;
				case IMG_8Bit:
					//Specifies that we are printing image. 
					inputImageDataCommand += "i";
					break;
				case PCXFlipped_8Bit:
					//Specifies that we are printing image. 
					inputImageDataCommand += "P";
					break;
				case PCX_8Bit:
					inputImageDataCommand += "p";
					break;
				default:
					inputImageDataCommand+= "B";
			}

			//Up to 16 characters used as an image name
			inputImageDataCommand += imageName + "\r\n";
		}
		return inputImageDataCommand;
	}
	
	/**
	   * <p>Right pad a String with a specified character.</p>
	   *
	   * <p>The String is padded to the size of <code>size</code>.</p>
	   *
	   * <pre>
	   * StringUtils.rightPad(null, *, *)     = null
	   * StringUtils.rightPad("", 3, 'z')     = "zzz"
	   * StringUtils.rightPad("bat", 3, 'z')  = "bat"
	   * StringUtils.rightPad("bat", 5, 'z')  = "batzz"
	   * StringUtils.rightPad("bat", 1, 'z')  = "bat"
	   * StringUtils.rightPad("bat", -1, 'z') = "bat"
	   * </pre>
	   *
	   * @param str  the String to pad out, may be null
	   * @param size  the size to pad to
	   * @param padChar  the character to pad with
	   * @return right padded String or original String if no padding is necessary,
	   *  <code>null</code> if null String input
	   * @since 2.0
	   */
	public static String rightPad(String str, int size, char padChar) 
	{
	      if (str == null) {
	          return null;
	      }
	      int pads = size - str.length();
	      if (pads <= 0) {
	          return str; // returns original String when possible
	      }
	      if (pads > 8192) {
	          return rightPad(str, size, String.valueOf(padChar));
	      }
	      return str.concat(padding(pads, padChar));
	  }
	 /**
	   * <p>Right pad a String with a specified String.</p>
	   *
	   * <p>The String is padded to the size of <code>size</code>.</p>
	   *
	   * <pre>
	   * StringUtils.rightPad(null, *, *)      = null
	   * StringUtils.rightPad("", 3, "z")      = "zzz"
	   * StringUtils.rightPad("bat", 3, "yz")  = "bat"
	   * StringUtils.rightPad("bat", 5, "yz")  = "batyz"
	   * StringUtils.rightPad("bat", 8, "yz")  = "batyzyzy"
	   * StringUtils.rightPad("bat", 1, "yz")  = "bat"
	   * StringUtils.rightPad("bat", -1, "yz") = "bat"
	   * StringUtils.rightPad("bat", 5, null)  = "bat  "
	   * StringUtils.rightPad("bat", 5, "")    = "bat  "
	   * </pre>
	   *
	   * @param str  the String to pad out, may be null
	   * @param size  the size to pad to
	   * @param padStr  the String to pad with, null or empty treated as single space
	   * @return right padded String or original String if no padding is necessary,
	   *  <code>null</code> if null String input
	   */
	  public static String rightPad(String str, int size, String padStr) {
	      if (str == null) {
	          return null;
	      }
	      if (padStr.equals("")) {
	          padStr = " ";
	      }
	      int padLen = padStr.length();
	      int strLen = str.length();
	      int pads = size - strLen;
	      if (pads <= 0) {
	          return str; // returns original String when possible
	      }
	      if (padLen == 1 && pads <= 8192) {
	          return rightPad(str, size, padStr.charAt(0));
	      }

	      if (pads == padLen) {
	          return str.concat(padStr);
	      } else if (pads < padLen) {
	          return str.concat(padStr.substring(0, pads));
	      } else {
	          char[] padding = new char[pads];
	          char[] padChars = padStr.toCharArray();
	          for (int i = 0; i < pads; i++) {
	              padding[i] = padChars[i % padLen];
	          }
	          return str.concat(new String(padding));
	      }
	  }
	  /**
	   * <p>Returns padding using the specified delimiter repeated
	   * to a given length.</p>
	   *
	   * <pre>
	   * StringUtils.padding(0, 'e')  = ""
	   * StringUtils.padding(3, 'e')  = "eee"
	   * StringUtils.padding(-2, 'e') = IndexOutOfBoundsException
	   * </pre>
	   *
	   * <p>Note: this method doesn't not support padding with
	   * <a href="http://www.unicode.org/glossary/#supplementary_character">Unicode Supplementary Characters</a>
	   * as they require a pair of <code>char</code>s to be represented.
	   * If you are needing to support full I18N of your applications
	   * consider using {@link #repeat(String, int)} instead. 
	   * </p>
	   *
	   * @param repeat  number of times to repeat delim
	   * @param padChar  character to repeat
	   * @return String with repeated character
	   * @throws IndexOutOfBoundsException if <code>repeat &lt; 0</code>
	   * 
	   */
	  private static String padding(int repeat, char padChar) throws IndexOutOfBoundsException {
	      if (repeat < 0) {
	          throw new IndexOutOfBoundsException("Cannot pad a negative amount: " + repeat);
	      }
	      final char[] buf = new char[repeat];
	      for (int i = 0; i < buf.length; i++) {
	          buf[i] = padChar;
	      }
	      return new String(buf);
	  }



}