package datamaxoneil.printer;

import java.io.ByteArrayOutputStream;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This is the base class for the O'Neil printer document objects. Document
 * objects hold your 'print commands' that you can send off to the PrinterONeil
 * object for printing. While the documents inner-workings are quite different
 * between the DocumentEZ and DocumentLP classes, giving them a single base
 * class will allow you to group them together in a single collection so they
 * can be sent to the printer in one large batch.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public abstract class Document {


	/**
	 * Rendering Enumerators These values will specify which type of printing is
	 * being done so the rendering engine call append/prepend the escape codes
	 * appropriate to this operation.
	 */
	protected enum PrintingType {
		/** Normal printing items like text */
		General,
		/** Standard 1D barcodes */
		Barcode,
		/** PDF417 2D barcode */
		Barcode417,
		/** Vertical and Horizontal lines  */
		Line,
		/** Graphic Images */
		Image,
		/** Graphics such as line, box, circle, polygon etc..*/
		Graphics
	}

	/** Printer Escape Code */
	public static char ESC = 0x1B;

	/** Printer EOL sequence */
	protected static byte[] EOL = { 0x0D, 0x0A };

	/** Rendered Document Object */
	protected ByteArrayOutputStream m_Document = new ByteArrayOutputStream(1024);

	/** The font to use when one is not specified. */
	private String m_DefaultFont = "";

	/** Expected length for a font name **/
	int m_FontNameLength = 0;

	/** Should print in Landscape mode. */
	private boolean m_LandscapeMode = false;

	/**
	 * The default font is specified in the construction of this object and is
	 * used when a font is not specified by the optional 'Parameters' parameter.
	 * 
	 * @return The printer's name for this font.
	 * @since 1.0.0
	 */
	public String getDefaultFont() {
		return m_DefaultFont;
	}

	/**
	 * The default font is specified in the construction of this object and is
	 * used when a font is not specified by the optional 'Parameters' parameter.
	 * This function is called by the constructors and is not accessible by
	 * others.
	 * 
	 * @param fontName The printer's name for this font.
	 * @since 1.1.0
	 */
	public void setDefaultFont(String fontName) {

		// Check inputs
		if ((fontName == null) || (fontName.length() != m_FontNameLength)) {
			// Out of range
			throw new IllegalArgumentException("Parameter 'fontName' must be " + m_FontNameLength
					+ " characters in length.");
		}

		// Assign Value
		m_DefaultFont = fontName;

		return;
	}

	/**
	 * Landscape mode causes the printing to be rotated by 90 degrees clockwise
	 * so the entire document is printed in landscape mode. The default value
	 * for this setting is false.
	 * 
	 * @return The landscape mode setting, true if enabled false otherwise.
	 * @since 1.1.0
	 */
	public boolean getIsLandscapeMode() {
		return m_LandscapeMode;
	}

	/**
	 * Landscape mode causes the printing to be rotated by 90 degrees clockwise
	 * so the entire document is printed in landscape mode. The default value
	 * for this setting is false.
	 * 
	 * @param isLandscapeMode The landscape mode setting, true if enabled false
	 *        otherwise.
	 * @since 1.1.0
	 */
	public void setIsLandscapeMode(boolean isLandscapeMode) {
		m_LandscapeMode = isLandscapeMode;

		return;
	}

	/**
	 * This method is used to clear the internal document contents that were
	 * written to it. This will not change the global parameter settings given
	 * on object creation.
	 * 
	 * @since 1.0.0
	 */
	public void clear() {
		m_Document.reset();

		return;
	}

	/**
	 * This method will write the text string to the Document byte array. This
	 * should only be used for pure ASCII 0x20 - 0x7F characters or who knows
	 * what it will convert it into.
	 * 
	 * @param dataStream Document object data stream. 
	 * @param text Text to add to the document.
	 * @since 1.1.0
	 */
	protected void addToDoc(ByteArrayOutputStream dataStream, String text) {
		char[] charArray = text.toCharArray();

		for (int index = 0; index < charArray.length; index++) {
			if (charArray[index] <= 0xFF) {
				// Single Byte
				addToDoc(dataStream, (byte) charArray[index]);
			}
			else {
				// Double Byte
				addToDoc(dataStream, (byte) (charArray[index] >> 8));
				addToDoc(dataStream, (byte) (charArray[index] & 0xFF));
			}
		}

		return;
	}

	/**
	 * This method will write the byte array to the Document byte array.
	 * 
	 * @param dataStream Document object data stream.
	 * @param data Bytes to add to the document.
	 * @since 1.1.0
	 */
	protected void addToDoc(ByteArrayOutputStream dataStream, byte[] data) {
		try {
			dataStream.write(data);
		}
		catch (java.io.IOException ignore) {
		}
		return;
	}
	/**
	 * This method checks whether the specified string matches the
	 * specified pattern
	 * @param input - string to check
	 * @param regex - pattern to match
	 * @return true if input matches pattern. false if input does not match pattern.
	 */
	public static boolean matches(CharSequence input,String regex) {
	    Pattern p = Pattern.compile(regex);
	    Matcher m = p.matcher(input);
	    return m.matches();
	}

	/**
	 * This method will write the byte to the Document byte array.
	 * 
	 * @param dataStream Document object data stream.
	 * @param data Byte to add to the document.
	 * @since 1.1.0
	 */
	protected void addToDoc(ByteArrayOutputStream dataStream, byte data) {
		dataStream.write(data);
		return;
	}

	/**
	 * This method will return the rendered document in the printer's language
	 * for sending directly to the printer.
	 * 
	 * @return Rendered document in the printer's language.</returns>
	 * @since 1.1.0
	 */
	public abstract byte[] getDocumentData();
}
