package datamaxoneil.connection;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import datamaxoneil.Monitor;

/**
 * This is the class for TCP type connections. The Transfer Control
 * Protocol(TCP) sends its data grams to the target device and does guarantee
 * reliable and in-order delivery. Because of this it is much slower than UDP so
 * is used when accurate data is more important than speed.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class Connection_TCP extends ConnectionBase {

	/** TCP Listener Object */
	private ServerSocket m_TcpListener = null;

	/** TCP Client Object */
	private Socket m_TcpClient = null;

	/** Stream to read from the TCP device */
	private DataInputStream m_StreamRead = null;

	/** Stream to read from the TCP device */
	private DataOutputStream m_StreamWrite = null;

	/** Target Address */
	private String m_Address = "127.0.0.1";

	/** Target Port */
	private int m_Port = 515;

	/** This will return the IP of the other end.
	 *  
	 * @return IP of the other end of the connection.
	 */
	public String getRemoteEnd() {
		return (m_TcpClient == null) ? "-none-" : m_TcpClient.getRemoteSocketAddress().toString();
	}

	/**
	 * We only want connections to be able to be created from our static
	 * methods. This will create a TCP connection object from the provided
	 * parameters.
	 * 
	 * @param isServer Will this be in server mode.
	 * @param targetDevice IP or address of target address.
	 * @param port Port number to connect on.
	 */
	protected Connection_TCP(boolean isServer, String targetDevice, int port) {
		super(isServer);

		// Setup Connection
		m_Port = port;
		m_Address = targetDevice;

		return;
	}

	/**
	 * Create a TCP client connection. This function is used to create a
	 * connection to a TCP server.
	 * 
	 * @param targetDevice IP or address of target address.
	 * @param port Port number to connect on.
	 * @return Connection object that can be used to talk to the device.
	 */
	static public Connection_TCP createClient(String targetDevice, int port) {
		return new Connection_TCP(false, targetDevice, port);
	}

	/**
	 * Create a TCP server connection. This function is used to create a
	 * connection which will listen for incoming TCP clients.
	 * 
	 * @param port Port number to connect on.
	 * @return Connection object that can be used to talk to the device.
	 */
	static public Connection_TCP createServer(int port) {
		return new Connection_TCP(true, "", port);
	}

	/**
	 * This will return true if there is data available to be read. This
	 * indicates if the device itself, however we are talking to it, has data to
	 * read.
	 */
	@Override
	protected boolean getHasData() {
		boolean hasData = false;

		try {
			hasData = (m_StreamRead.available() > 0);
		}
		catch (Exception oops) {
		}

		return hasData;
	}

	/**
	 * This will open the current connection if not open. The base class
	 * function handles the starting of the threads used to do the asynchronous
	 * communication with the device. The derived class will do the device
	 * specific routines.
	 */
	@Override
	protected boolean innerOpen() throws IOException {

		// Try to open
		if (getIsServerMode()) {
			// Server
			if (m_Reconnecting == false) {
				// If reconnecting already have a listener
				m_TcpListener = new ServerSocket(m_Port);
			}

			// Mark as open
			m_IsOpen = true;
		}
		else {
			// Client
			m_TcpClient = new Socket(m_Address, m_Port);
			m_StreamRead = new DataInputStream(m_TcpClient.getInputStream());
			m_StreamWrite = new DataOutputStream(m_TcpClient.getOutputStream());

			m_IsOpen = (m_StreamRead != null) && (m_StreamWrite != null);
			m_IsActive = m_IsOpen;
		}

		return m_IsOpen;
	}

	/**
	 * This will close the current connection, if open, after all existing items
	 * have finished printing. This is the internally called function that is
	 * called with the parameter as true if it is used within this framework. If
	 * the user calls Close, then that Close will pass a false. This will allow
	 * is to wait for the thread to finish when the user initiates it.
	 */
	@Override
	protected void close(boolean isInternalCall) {
		int timeout = (isInternalCall) ? 0 : Integer.MAX_VALUE;

		do {
			if (Monitor.tryEnter(m_LockGeneral, 0)) {
				try {
					// We acquired a lock
					timeout = 0;

					if (getIsOpen()) {
						// Base Class
						closeBase(isInternalCall);

						// Close
						try {
							if (m_StreamRead != null)
								m_StreamRead.close();
						}
						catch (Exception oops) {
						}
						try {
							if (m_StreamWrite != null)
								m_StreamWrite.close();
						}
						catch (Exception oops) {
						}
						try {
							if (m_TcpClient != null)
								m_TcpClient.close();
						}
						catch (Exception oops) {
						}
						try {
							if ((m_Reconnecting == false) && (m_TcpListener != null))
								m_TcpListener.close();
						}
						catch (Exception oops) {
						}

						// Clear
						m_StreamRead = null;
						m_StreamWrite = null;
						m_TcpClient = null;
						if (m_Reconnecting == false)
							m_TcpListener = null;

						// Adjust state values
						m_IsOpen = false;
						m_IsActive = false;
					}
				}
				catch (Exception oops) {
				}
				finally {
					Monitor.exit(m_LockGeneral);
				}
			}
			else {
				// Decrement Timeout
				timeout -= 100;
				if (timeout > 0) {
					try {
						Thread.sleep(100);
					}
					catch (Exception ignore) {
					}
				}
			}
		}
		while (timeout > 0);

		return;
	}

	/**
	 * This will read the existing data from the device creating the byte array
	 * object and returning the number of bytes in the array.
	 */
	@Override
	protected int innerRead(byte[] buffer) throws IOException {
		return m_StreamRead.read(buffer);
	}

	/**
	 * This will write the provided buffer to the device.
	 */
	@Override
	protected void innerWrite(byte[] buffer) throws IOException {
		m_StreamWrite.write(buffer);
		m_StreamWrite.flush();

		return;
	}

	/**
	 * This will return true if we have connected to a new client.
	 */
	@Override
	protected boolean innerListen() {
		boolean hasConnection = false;

		// Check for pending connections
		try {
			m_TcpClient = m_TcpListener.accept();
			m_StreamRead = new DataInputStream(m_TcpClient.getInputStream());
			m_StreamWrite = new DataOutputStream(m_TcpClient.getOutputStream());
			hasConnection = true;
		}
		catch (Exception ignore) {
		}

		return hasConnection;
	}

	/**
	 * A summary of the configuration.
	 */
	@Override
	protected String configSummary() {
		String results = "";

		// Build
		results = "TCP " + ((getIsServerMode()) ? "(Server)" : "(Client)");

		return results;
	}

	/**
	 * A single line description of the configuration.
	 */
	@Override
	protected String configCompact() {
		String results = "";

		// Build
		results = m_TcpClient.getRemoteSocketAddress() + ": " + m_TcpClient.getPort();

		return results;
	}
	
	/**
	 * A detailed description of the configuration.
	 */
	@Override
	protected String configDetail() {
		String results = "";

		// Build
		if (getIsServerMode()) {
			// Ethernet Server
			results += "" + "TCP Server Settings\r\n" + "   Port: " + m_Port;
		}
		else if (getIsClientMode()) {
			// Ethernet Client
			results += "" + "TCP Client Settings\r\n" + "Target IP: "
					+ m_TcpClient.getRemoteSocketAddress() + "\r\n" + "Port: "
					+ m_TcpClient.getPort();
		}

		return results;
	}
}