package datamaxoneil.connection;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import datamaxoneil.Monitor;
import datamaxoneil.printer.configuration.GeneralConfiguration.Handshake;
import datamaxoneil.printer.configuration.GeneralConfiguration.Parity;
import gnu.io.PortInUseException;
import gnu.io.RXTXPort;
import gnu.io.SerialPort;
import gnu.io.UnsupportedCommOperationException;

/**
 * This is the class for Serial type connections.
 * A serial connection, also known as RS232, sends its data over a serial
 * port.  In some instances Bluetooth connections can be 'mapped' to
 * a serial port and this class would then be used to communicate with it.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class Connection_Serial extends ConnectionBase {
	
	/** Parity options for the connection */
	protected enum RXTX_Parity {
		/** No control is used for the handshake */
		None(SerialPort.PARITY_NONE, "None", Parity.None), 
		/** Sets the parity bit so that the count of bits set is an odd number */
		Odd(SerialPort.PARITY_ODD, "Odd", Parity.Odd),
		/** Sets the parity bit so that the count of bits set is an even number */
		Even(SerialPort.PARITY_EVEN, "Even", Parity.Even);
				
		/** External Value */
		private final int m_Value;
		
		/** User Friendly Name */
		private final String m_Name;
		
		/** Data value in local ENum type */
		private final Parity m_LocalValue;
		
		/**
		 * Constructor
		 * @param value External Value
		 * @param name User Friendly Name
		 * @param localValue Data value in local ENum type
		 */
		RXTX_Parity(int value, String name, Parity localValue) {
			m_Value = value; 
			m_Name = name; 
			m_LocalValue = localValue;}	
				
		/**
		 * External Value
		 * @return External Value
		 */
		public int value() { return m_Value; }
				
		/**
		 * User Friendly Name
		 * @return User Friendly Name
		 */
		public String named() { return m_Name; }
		
		/**
		 * Data value in local ENum type
		 * @return Data value in local ENum type
		 */
		public Parity localValue() { return m_LocalValue; }
	};
	
	/** Handshaking options for the connection */
	protected enum RXTX_Handshake {
		/** No control is used for the handshake */
		None(SerialPort.FLOWCONTROL_NONE, "None", Handshake.None),
		/** The XON/XOFF software control protocol is used. The XOFF control is sent to stop the transmission of data. The XON control is sent to resume the transmission. These software controls are used instead of Request to Send (RTS) and Clear to Send (CTS) hardware controls */
		Software(SerialPort.FLOWCONTROL_XONXOFF_IN, "Software", Handshake.Software),
		/** Request-to-Send (RTS) hardware flow control is used. RTS signals that data is available for transmission. If the input buffer becomes full, the RTS line will be set to false. The RTS line will be set to true when more room becomes available in the input buffer */
		Hardware(SerialPort.FLOWCONTROL_RTSCTS_IN, "Hardware", Handshake.Hardware),
		/** Both the Request-to-Send (RTS) hardware control and the XON/XOFF software controls are used */
		Both(SerialPort.FLOWCONTROL_XONXOFF_IN | SerialPort.FLOWCONTROL_RTSCTS_IN, "Both", Handshake.Both);
	
		/** External Value */
		private final int m_Value;
				
		/** User Friendly Name */
		private final String m_Name;
		
		/** Data value in local ENum type */
		private final Handshake m_LocalValue;
		
		/**
		 * Constructor
		 * @param value External Value
		 * @param name User Friendly Name
		 * @param localValue Data value in local ENum type
		 */
		RXTX_Handshake(int value, String name, Handshake localValue) { 
			m_Value = value; 
			m_Name = name; 
			m_LocalValue = localValue; }		
		
		/**
		 * External Value
		 * @return External Value
		 */
		public int value() { return m_Value; }
				
		/**
		 * User Friendly Name
		 * @return User Friendly Name
		 */
		public String named() { return m_Name; }
				
		/**
		 * Data value in local ENum type
		 * @return Data value in local ENum type
		 */
		public Handshake localValue() { return m_LocalValue; }
	};
	
	/** Stop bit options for the connection */
	public enum StopBits { 
		
		/** One stop bit is used. */
		One(SerialPort.STOPBITS_1, "1"),
		/** Two stop bits are used.  */
		Two(SerialPort.STOPBITS_2, "2"), 
		/** 1.5 stop bits are used. */
		OnePointFive(SerialPort.STOPBITS_1_5, "1.5"); 
		
		/** External Value */
		private final int m_Value;
		
		/** User Friendly Name */
		private final String m_Name;
		
		/**
		 * Constructor
		 * @param value External Value
		 * @param name User Friendly Name
		 */
		StopBits(int value, String name) { m_Value = value; m_Name = name; }		
		
		/**
		 * External Value
		 * @return External Value
		 */
		public int value() { return m_Value; }
				
		/**
		 * User Friendly Name
		 * @return User Friendly Name
		 */
		public String named() { return m_Name; }
	};
	
	/** Connection Object */
	private RXTXPort m_Connection = null;

	/** Stream to read from the Serial device */
	private DataInputStream m_StreamRead = null;

	/** Stream to read from the Serial device */
	private DataOutputStream m_StreamWrite = null;

	/** Name of the serial port the connection is using. */
	private String m_Port;
	
	/** Baud rate the connection is using. */
	private int m_BaudRate;
	
	/** Parity the connection is using. */
	private RXTX_Parity m_Parity;
	
	/** Data bits the connection is using. */
	private int m_DataBits;
	
	/** Stop Bits the connection is using. */
	private StopBits m_StopBits;
	
	/** Handshaking the connection is using. */
	private RXTX_Handshake m_Handshaking;

	/** 
	 * Name of the serial port the connection is using.
	 * @return Port name
	 */	 
	public String getPortName() {
		return m_Port;
	}

	/**
	 * Baud rate the connection is using.
	 * @return Baud rate
	 */
	public int getPortBaud() {
		return m_BaudRate;
	}

	/**
	 * Parity the connection is using.
	 * @return Parity value
	 */
	public Parity getPortParity() {
		return m_Parity.localValue();
	}

	/**
	 * Data bits the connection is using.
	 * @return Data bits value
	 */
	public int getPortDataBits() {
		return m_DataBits;
	}

	/**
	 * Stop Bits the connection is using.
	 * @return Stop bits value
	 */
	public StopBits getPortStopBits() {
		return m_StopBits;
	}

	/**
	 * Handshaking the connection is using.
	 * @return Handshaking value
	 */
	public Handshake getHandshaking() {
		return m_Handshaking.localValue();
	}

	/**
	 * We only want connections to be able to be created from our static
	 * methods. This will create a TCP connection object from the provided
	 * parameters.
	 * 
	 * @param isServer Are we in server mode, or client.
	 * @param portName Name of the COM port.
	 * @param portBaud Baud to connect at.
	 * @param portParity Parity to connect with.
	 * @param portDataBits DataBits to connect with.
	 * @param portStopBits StopBits to connect with.
	 * @param portHandshaking Handshaking to connect with.
	 */
	protected Connection_Serial(boolean isServer, String portName, int portBaud,
			Parity portParity, int portDataBits, StopBits portStopBits,
			Handshake portHandshaking) {
		super(isServer);
		
		// Convert Parity and Handshaking
		if (portParity == RXTX_Parity.None.localValue()) {
			m_Parity = RXTX_Parity.None;
		}
		else if (portParity == RXTX_Parity.Even.localValue()) {
			m_Parity = RXTX_Parity.Even;
		}
		else if (portParity == RXTX_Parity.Odd.localValue()) {
			m_Parity = RXTX_Parity.Odd;
		}
		else {
			// Illegal Value
			throw new IllegalArgumentException("Partiy value is not valid.");
		}
		
		if (portHandshaking == RXTX_Handshake.None.localValue()) {
			m_Handshaking = RXTX_Handshake.None;
		}
		else if (portHandshaking == RXTX_Handshake.Hardware.localValue()) {
			m_Handshaking = RXTX_Handshake.Hardware;
		}
		else if (portHandshaking == RXTX_Handshake.Software.localValue()) {
			m_Handshaking = RXTX_Handshake.Software;
		}
		else if (portHandshaking == RXTX_Handshake.Both.localValue()) {
			m_Handshaking = RXTX_Handshake.Both;
		}
		else {
			// Illegal Value
			throw new IllegalArgumentException("Handshake value is not valid.");
		}

		// Setup Connection
		m_Port = portName;
		m_BaudRate = portBaud;
		m_DataBits = portDataBits;
		m_StopBits = portStopBits;

		return;
	}

	/**
	 * Create a serial client connection.
	 * 
	 * @param portName Name of the COM port.
	 * @param portBaud Baud to connect at.
	 * @param portParity Parity to connect with.
	 * @param portDataBits DataBits to connect with.
	 * @param portStopBits StopBits to connect with.
	 * @param portHandshaking Handshaking to connect with.
	 * @return Connection object that can be used to talk to the device.
	 */
	static public Connection_Serial createClient(String portName, int portBaud, Parity portParity, int portDataBits, StopBits portStopBits, Handshake portHandshaking) { 
		return new Connection_Serial(false, portName, portBaud, portParity, portDataBits, portStopBits, portHandshaking);
	}

	/**
	 * Create a serial server connection.
	 * 
	 * @param portName Name of the COM port.
	 * @param portBaud Baud to connect at.
	 * @param portParity Parity to connect with.
	 * @param portDataBits DataBits to connect with.
	 * @param portStopBits StopBits to connect with.
	 * @param portHandshaking Handshaking to connect with.
	 * @return Connection object that can be used to talk to the device.
	 */
	static public Connection_Serial createServer(String portName, int portBaud, Parity portParity, int portDataBits, StopBits portStopBits, Handshake portHandshaking) { 
		return new Connection_Serial(true, portName, portBaud, portParity, portDataBits, portStopBits, portHandshaking);
	}

	/**
	 * This will return true if there is data available to be read. This
	 * indicates if the device itself, however we are talking to it, has data to
	 * read.
	 * 
	 * @return True if there is data available to be read.
	 */
	@Override
	protected boolean getHasData() {
		boolean hasData = false;

		try {
			hasData = (m_StreamRead.available() > 0);
		}
		catch (Exception oops) {
		}

		return hasData;
	}

	/**
	 * This will open the current connection if not open. The base class
	 * function handles the starting of the threads used to do the asynchronous
	 * communication with the device. The derived class will do the device
	 * specific routines.
	 */
	@Override
	public boolean innerOpen() throws IOException {
				
		// Try to open
		try {
			// Client & Server same for serial
			m_Connection = new RXTXPort(m_Port);
			m_Connection.setSerialPortParams(
					m_BaudRate,
					m_DataBits,
					m_StopBits.value(),
					m_Parity.value());
			m_Connection.setFlowControlMode(m_Handshaking.value());
			
			m_StreamRead = new DataInputStream(m_Connection.getInputStream());
			m_StreamWrite = new DataOutputStream(m_Connection.getOutputStream());

			m_IsOpen = (m_StreamRead != null) && (m_StreamWrite != null);
			m_IsActive = m_IsOpen;
		}
		catch (PortInUseException oops) {
			throw new IOException(oops.getMessage());
		}
		catch (UnsupportedCommOperationException oops) {
			throw new IOException(oops.getMessage());
		}
		
		return m_IsOpen;
	}

	/**
	 * This will close the current connection, if open, after all existing items
	 * have finished printing. This is the internally called function that is
	 * called with the parameter as true if it is used within this framework. If
	 * the user calls Close, then that Close will pass a false. This will allow
	 * is to wait for the thread to finish when the user initiates it.
	 */
	@Override
	protected void close(boolean isInternalCall) {
		int timeout = (isInternalCall) ? 0 : Integer.MAX_VALUE;

		do {
			if (Monitor.tryEnter(m_LockGeneral, 0)) {
				try {
					// We acquired a lock
					timeout = 0;

					if (getIsOpen()) {
						// Base Class
						closeBase(isInternalCall);

						// Close
						try {
							if (m_StreamRead != null)
								m_StreamRead.close();
						}
						catch (Exception oops) {
						}
						try {
							if (m_StreamWrite != null)
								m_StreamWrite.close();
						}
						catch (Exception oops) {
						}
						try {
							if (m_Connection != null)
								m_Connection.close();
						}
						catch (Exception oops) {
						}

						// Clear
						m_StreamRead = null;
						m_StreamWrite = null;
						m_Connection = null;

						// Adjust state values
						m_IsOpen = false;
						m_IsActive = false;
					}
				}
				catch (Exception oops) {
				}
				finally {
					Monitor.exit(m_LockGeneral);
				}
			}
			else {
				// Decrement Timeout
				timeout -= 100;
				if (timeout > 0) {
					try {
						Thread.sleep(100);
					}
					catch (Exception ignore) {
					}
				}
			}
		}
		while (timeout > 0);

		return;
	}

	/**
	 * This will read the existing data from the device creating the byte array
	 * object and returning the number of bytes in the array.
	 */
	@Override
	protected int innerRead(byte[] buffer) throws IOException {
		return m_StreamRead.read(buffer);
	}

	/**
	 * This will write the provided buffer to the device.
	 */
	@Override
	protected void innerWrite(byte[] buffer) throws IOException {
		m_StreamWrite.write(buffer);
		m_StreamWrite.flush();

		return;
	}

	/**
	 * This will return true if we have connected to a new client.
	 */
	@Override
	protected boolean innerListen() {
		return true;
	}

	/**
	 * A summary of the configuration.
	 */
	@Override
	protected String configSummary() {
		String results = "";

		// Build
		results = "Serial " + ((getIsServerMode()) ? "(Server)" : "(Client)");

		return results;
	}

	/**
	 * A single line description of the configuration.
	 */
	@Override
	protected String configCompact() {
		String results = "";

		// Build
		results = getPortName() + ": " + getPortBaud();

		return results;
	}
	
	/**
	 * A detailed description of the configuration.
	 */
	@Override
	protected String configDetail() {
		String results = "";

		// Build
		results += configSummary() + "\r\n" +
			"   Comm Port: " + getPortName() + "\r\n" +
			"   Baud Rate: " + getPortBaud() + "\r\n" +
			"   Parity: " + getPortParity().name() + "\r\n" +
			"   Data Bits: " + getPortDataBits() + "\r\n" +
			"   Stop Bits: " + getPortStopBits() + "\r\n" +
			"   Handshake: " + getHandshaking().name();
		
		return results;
	}
}