package datamaxoneil.connection;

import java.util.ArrayList;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import datamaxoneil.Monitor;

/**
 * This class encapsulates all of the different connection methods into a single
 * uniform base class. The main purpose of this set of classes is for
 * communication with an ONeil printer, but it can be used to connect to any
 * device that supports the protocols here in. Using the ConnectionBase object
 * type is a good way to make a generalized set of routines that can accept any
 * type of connection.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
abstract public class ConnectionBase extends Thread {

	/**
	 * These are the different responses from the printer. These responses are
	 * sent in response to special configuration commands that can be sent to
	 * the printer. After these commands are sent the specific printer response
	 * can queried for with WaitResponse() translate them into human readable
	 * values.
	 */
	public enum PrinterResponse {

		/** The result was either unrecognized or never set. */
		Unknown,

		/** The printer returned that it acknowledged the command. */
		ACK,

		/** Error last data transferred was invalid */
		NAK,

		/** The printer is writing the last command. */
		WRITING,

		/** The printer is ready for the next block in the transfer. */
		RESUME,

		/** The transfer has been successfully completed. */
		DONE
	};

	/**
	 * General internal lock to keep all the Open() and Close operations thread
	 * safe. This lock is used in the Open and Close operations for this class
	 * and subclasses to make sure we are not trying to Open and Close a
	 * connection simultaneously.
	 */
	protected Object m_LockGeneral = new Object();

	/**
	 * Thread lock for receive operations. This is a specialized lock for ensuring
	 * operations that work on receiving objects are thread safe.
	 */
	private Object m_DataLockRecv = new Object();

	/**
	 * Thread lock for send operations. This is a specialized lock for ensuring
	 * operations that work on sending objects are thread safe.
	 */
	private Object m_DataLockSend = new Object();

	/**
	 * Background thread that handles all of the connections, reading and
	 * writing so all operations can be performed asynchronously.
	 */
	private volatile boolean m_WorkerThreadActive = false;

	/**
	 * MemoryStream to store received data in. As data is read from the device
	 * it will be stored in here. When the user requests a read the data will be
	 * taken from the start of this stream. All of the unread data will then be
	 * shifted to the beginning of the stream and the read and write pointers
	 * are adjusted to match.
	 */
	private ByteArrayOutputStream m_mStreamRecv = new ByteArrayOutputStream(1024);

	/**
	 * Are we a server (or a client). During the startup process clients simply
	 * connect to their target while servers must listen for incoming
	 * connections and then connect to it when it arrives.
	 */
	private boolean m_IsServerMode = false;

	/**
	 * Should we try to reconnect. If we are connected to a device as a client
	 * and the connection drops we will try to reconnect once. This flag is set
	 * to true during this process.
	 */
	protected boolean m_Reconnecting = false;

	/**
	 * These hold the data to send. Data to be sent to the target device is
	 * converted into a byte array and then stored here. The data is then sent
	 * by the background thread.
	 */
	private ArrayList<byte[]> m_DataStorageSend = new ArrayList<byte[]>();

	/**
	 * Is the connection open. This is true when the connection is open or
	 * listening and false if we have never connected or the connection is
	 * closed.
	 */
	protected boolean m_IsOpen = false;

	/**
	 * Is the connection open and talking to a target device. For client
	 * connections this is the same as m_IsOpen, but for servers m_IsOpen means
	 * we are listening and m_IsActive means we are connected.
	 */
	protected boolean m_IsActive = false;

	/**
	 * This flag is set to true when close() is called. This will cause any more
	 * external writes, reads, and opens to fail.
	 */
	private boolean m_IsClosing = false;

	/**
	 * This flag indicates if clear was called during a write
	 */
	boolean m_ClearTriggered = false;

	/**
	 * Is the connection open. This is true when the connection is open or
	 * listening and false if we have never connected or the connection was
	 * closed.
	 * 
	 * @return True when the connection is open or listening.
	 */
	public boolean getIsOpen() {
		return m_IsOpen | m_Reconnecting;
	}

	/**
	 * Is the connection open and talking to a target device. For client
	 * connections this is the same as isOpen(), but for servers
	 * isOpen means we are listening and isActive means we are
	 * connected.
	 * 
	 * @return True when connection open and talking to a target device.
	 */
	public boolean getIsActive() {
		return m_IsActive;
	}
	
	/**
	 * Specifies whether to close the connection or not.Use this to close connection instead of close().
	 * @param isClosing - true to close connection. false to keep connection open()
	 */
	public void setIsClosing(boolean isClosing)
	{
		m_IsClosing = isClosing;
	}

	/**
	 * This will return the number of bytes that are ready to be read from the
	 * device. Data received from the target device is read asynchronously and
	 * stored in an internal buffer. So this value is the number of bytes that
	 * have been received and are being stored locally.
	 * 
	 * @return The number of bytes that are ready to be read. 
	 */
	public long getBytesAvailable() {
		long length;

		// Synchronization
		synchronized (m_DataLockRecv) {
			length = m_mStreamRecv.size();
		}

		return length;
	}

	/**
	 * Are we in server mode. This is true if the current connection is for a
	 * server or false if it is for a client.
	 * 
	 * @return Are we in server mode.
	 */
	public boolean getIsServerMode() {
		return m_IsServerMode;
	}

	/**
	 * Are we in client mode. This is true if the current connection is for a
	 * client or false if it is for a server.
	 * 
	 * @return Are we in client mode.
	 */
	public boolean getIsClientMode() {
		return !m_IsServerMode;
	}

	/**
	 * The base constructor simply configures the object as a client or a server
	 * by setting the m_IsServerMode parameter and creates the
	 * reader and writer associated with the receiving of data from the target
	 * device.
	 * 
	 * @param isServer Will this be in server mode.
	 */
	protected ConnectionBase(boolean isServer) {

		// Setup Server Mode
		m_IsServerMode = isServer;

		// In Java always have a thread existing
		setName("Communications");
		setDaemon(true);
		start();
		
		return;
	}

	/**
	 * Called by the garbage collector on an object when garbage collection 
	 * determines that there are no more references to the object. A subclass
	 * overrides the finalize method to dispose of system resources or to
	 * perform other cleanup.
	 *   
	 * @see java.lang.Object#finalize()
	 */
	@Override
	protected void finalize() throws Throwable {
		// TODO 
		System.out.println("finalize called");		
		return;
	}
	
	/**
	 * This will Clear the data waiting in the read buffer.
	 */
	public void clearReadBuffer() {

		synchronized (m_DataLockRecv) {
			// Move Position to start
			m_mStreamRecv.reset();
		}

		return;
	}

	/**
	 * This will Clear the data waiting in the write buffer.
	 */
	public void clearWriteBuffer() {

		synchronized (m_DataLockSend) {
			// Convert to byte array
			m_DataStorageSend.clear();
		}

		return;
	}

	/**
	 * This will return all of the current data read from the device in the form
	 * of a string.
	 * 
	 * @return All of the current data read from the device in the form
	 * of a string.
	 */
	public String read() {
		return read("", 0);
	}

	/**
	 * This will read data from the connection returning up to and including the
	 * end sequence is found or after the wait from the last bit of data is
	 * equal to or greater than the timeout.
	 * 
	 * @param endSequence String that marks the end of the data stream.
	 * @param msecTimeout Maximum amount of time to wait for data.
	 * @return The string returned.
	 */
	public String read(String endSequence, long msecTimeout) {
		int size;
		byte[] data;
		String dataRead = "";
		char[] charArray;
		long timeout = msecTimeout;

		// Check that we are closed or closing
		if (getIsOpen() == false) {
			// No Client
			throw new UnsupportedOperationException("Can not perform a Read on a Closed connection.");
		}

		if (m_IsClosing == true) {
			// We are in the process of closing
			throw new UnsupportedOperationException("Can not perform a Read while the connection is in the process of being Closed.");
		}

		// Synchronization
		if (timeout == 0) {
			// Instant method
			synchronized (m_DataLockRecv) {
				// Alter Stream
				size = m_mStreamRecv.size();

				// Check for emptiness
				if (size == 0) {
					// We are so alone
					return "";
				}

				// Move Position to start (this reads the whole thing)
				data = m_mStreamRecv.toByteArray();
				m_mStreamRecv.reset();
			}

			// Do a straight byte->char cast
			charArray = new char[size];
			for (int index = 0; index < size; index++) {
				charArray[index] = (char) data[index];
			}
			dataRead = new String(charArray);
		}
		else {
			// Timeout Version
			boolean foundFlag = false;
			boolean absoluteMode = (timeout < 0);
			long lastTime = System.currentTimeMillis();
			timeout = Math.abs(timeout);

			while (!foundFlag && ((System.currentTimeMillis() - lastTime) < timeout)) {
				// Check for emptiness
				if (m_mStreamRecv.size() == 0) {
					// We are so alone
					try {
						Thread.sleep(100);
					}
					catch (InterruptedException ignore) {
					}
					continue;
				}

				synchronized (m_DataLockRecv) {
					// Move Position to start (this reads the whole thing)
					size = m_mStreamRecv.size();
					data = m_mStreamRecv.toByteArray();
					m_mStreamRecv.reset();
				}

				// Do a straight byte->char cast
				charArray = new char[size];
				for (int index = 0; index < size; index++) {
					charArray[index] = (char) data[index];
				}
				dataRead += new String(charArray);

				// Update Found flag and time
				foundFlag = (dataRead.indexOf(endSequence) != -1);
				if (!absoluteMode)
					lastTime = System.currentTimeMillis();
			}
		}

		return dataRead;
	}

	/**
	 * This will write the current data read from the device into the buffer up
	 * to the provided length starting at the offset.
	 * 
	 * @param buffer Buffer to write data into.
	 * @param offset Index of where to start writing into the buffer.
	 * @param length Maximum amount of data to read into buffer.
	 * @return The number of bytes read.
	 */
	public int read(byte[] buffer, int offset, int length) {
		int lengthRead = 0;

		// Check that we are closed or closing
		if (getIsOpen() == false) {
			// No Client
			throw new UnsupportedOperationException("Can not perform a Read on a Closed connection.");
		}

		if (m_IsClosing == true) {
			// We are in the process of closing
			throw new UnsupportedOperationException("Can not perform a Read while the connection is in the process of being Closed.");
		}

		// Synchronization
		synchronized (m_DataLockRecv) {
			// Check for emptiness
			if (m_mStreamRecv.size() == 0) {
				// We are so alone
				return 0;
			}

			// Data Array
			byte[] data = m_mStreamRecv.toByteArray();

			// Adjust stream to move unread data to the start
			if (m_mStreamRecv.size() <= length) {
				// We read the whole thing
				lengthRead = data.length;
				System.arraycopy(data, 0, buffer, offset, lengthRead);

				m_mStreamRecv.reset();
			}
			else {
				// There was still some left
				lengthRead = length;
				System.arraycopy(data, 0, buffer, offset, lengthRead);

				m_mStreamRecv.reset();
				m_mStreamRecv.write(data, length, data.length - lengthRead);
			}
		}

		return lengthRead;
	}

	/**
	 * This will write the provided String to the device.
	 * 
	 * @param data Text string to write.
	 */
	public void write(String data) {
		byte[] dataArray = data.getBytes();
		write(dataArray, 0, dataArray.length);
		return;
	}

	/**
	 * This will write the data to the device from buffer.
	 * 
	 * @param buffer Buffer to read data from.
	 */
	public void write(byte[] buffer) {
		write(buffer, 0, buffer.length);
		return;
	}
	
	/**
	 * This will write the data to the device from buffer starting at the given
	 * offset up to the length indicated.
	 * 
	 * @param buffer Buffer to read data from.
	 * @param offset Index of where to start reading from the buffer.
	 * @param length Amount of data to read from buffer.
	 */
	public void write(byte[] buffer, int offset, int length) {
		byte[] copy;
		
		// Handle null like an empty buffer
		if (buffer == null)
			copy = new byte[0];
		else
			copy = new byte[Math.min(buffer.length - offset, length)];
		
		// Check that we are closed or closing
		if (getIsOpen() == false) {
			// No Client
			throw new UnsupportedOperationException("Can not perform a Write on a Closed connection.");
		}

		if (m_IsClosing == true) {
			// We are in the process of closing
			throw new UnsupportedOperationException("Can not perform a Write while the connection is in the process of being Closed.");
		}

		// Copy the data to our new buffer
		for (int L1 = 0; L1 < copy.length; L1++)
			copy[L1] = buffer[L1 + offset];

		// Synchronization
		synchronized (m_DataLockSend) {
			m_DataStorageSend.add(copy);
		}

		return;
	}

	/**
	 * Wait for a status response from the printer. This type of response is
	 * returned due to certain state or configuration commands and is not
	 * normally experienced by a normal use.
	 * 
	 * @param msecTimeout Maximum time to wait for response.
	 * @return Response returned from the printer.
	 */
	public PrinterResponse waitResponse(int msecTimeout) {
		String buffer = "";
		byte[] readBuffer = new byte[1];
		long time = System.currentTimeMillis();

		// Check that we are closed or closing
		if (getIsOpen() == false) {
			// No Client
			throw new UnsupportedOperationException("Can not perform a WaitResponse on a Closed connection.");
		}

		if (m_IsClosing == true) {
			// We are in the process of closing
			throw new UnsupportedOperationException("Can not perform a WaitResponse while the connection is in the process of being Closed.");
		}

		while ((System.currentTimeMillis() - time) < msecTimeout) {
			// Read
			if (read(readBuffer, 0, 1) == 0) {
				// Nothing so try again
				try {
					Thread.sleep(200);
				}
				catch (InterruptedException ignore) {
				}
				continue;
			}

			// Append
			buffer = (buffer.length() == 5) ? buffer.substring(1) + (char) readBuffer[0] : buffer
					+ (char) readBuffer[0];

			// Trim Buffer to start with {
			if (buffer.indexOf("{") != -1)
				buffer = buffer.substring(buffer.indexOf("{"));

			// Continue to read?
			if (buffer.length() < 5)
				continue;

			// Check
			if ((buffer.length() == 5) && (buffer.startsWith("{")) && (buffer.endsWith("}"))) {
				// We got our response
				PrinterResponse result = PrinterResponse.Unknown;

				if (buffer.startsWith("{A")) {
					// Ack
					result = PrinterResponse.ACK;
				}
				else if (buffer.startsWith("{N")) {
					// Nak
					result = PrinterResponse.NAK;
				}
				else if (buffer.startsWith("{W")) {
					// Writing
					result = PrinterResponse.WRITING;
				}
				else if (buffer.startsWith("{R")) {
					// Resume
					result = PrinterResponse.RESUME;
				}
				else if (buffer.startsWith("{D")) {
					// Writing
					result = PrinterResponse.DONE;
				}

				return result;
			}
		}

		return PrinterResponse.Unknown;
	}

	/**
	 * This will return true if there is data available to be read. This
	 * indicates if the device itself, however we are talking to it, has data to
	 * read.
	 * 
	 * @return True if there is data available to be read.
	 */
	abstract protected boolean getHasData();

	/**
	 * This will open the current connection if not open. The base class
	 * function handles the starting of the threads used to do the asynchronous
	 * communication with the device. The derived class will do the device
	 * specific routines.
	 * 
	 * @return Was the open successful.
	 * @throws IOException Exception thrown if open fails badly.
	 */
	public synchronized boolean open() throws IOException {
		boolean results = true;

		try {
			Monitor.enter(m_LockGeneral);

			// Call specific functionality
			if (m_IsOpen == false) {
				// Only open if not open (duh)
				results = innerOpen();

				// If open passed start the internals
				if (m_IsOpen) {
					// Run Worker Thread
					if (!m_WorkerThreadActive) {
						// If there is no worker thread then create one
						m_Reconnecting = false;
						m_IsClosing = false;
						m_WorkerThreadActive = true;
						notify();
					}

					// Clear Reconnect Flag
					m_Reconnecting = false;
				}
			}
		}
		finally {
			Monitor.exit(m_LockGeneral);
		}

		return results;
	}

	/**
	 * This will open the current connection if not open. The base class
	 * function handles the starting of the threads used to do the asynchronous
	 * communication with the device. The derived class will do the device
	 * specific routines.
	 * 
	 * @return Was the open successful.
	 * @throws IOException Exception thrown if open fails badly.
	 */
	protected abstract boolean innerOpen() throws IOException;

	/**
	 * This function will wait, up to the given timeout in msec, for the sending
	 * queue to be empty. The function will return true if the queue is empty,
	 * or false if it wasn't empty by the end of the time out.
	 * 
	 * @param timeout_msec Number of msec to wait for the sending queue to
	 * be empty.
	 * @return True if the queue was empty before the timeout.
	 */
	public boolean waitForEmptyBuffer(int timeout_msec) {
		long endTime = System.currentTimeMillis() + timeout_msec;

		// Check that we are closed or closing
		if (getIsActive() == false) {
			// No Client
			return true;
		}

		while (System.currentTimeMillis() < endTime) {
			// Check for empty buffer
			synchronized (m_DataLockSend) {
				if (m_DataStorageSend.size() == 0)
					return true;
			}

			// Still more so wait
			try {
				Thread.sleep(100);
			}
			catch (InterruptedException ignore) {
			}
		}

		return false;
	}

	/**
	 * This will close the current connection if open.
	 */
	public void close() {
		close(false);
		return;
	}

	/**
	 * This will close the current connection, if open, after all existing items
	 * have finished printing. This is the internally called function that is
	 * called with the parameter as true if it is used within this framework. If
	 * the user calls close(), then that Close will pass a false.
	 * This will allow is to wait for the thread to finish when the user
	 * initiates it.
	 * 
	 * @param isInternalCall Is this function being called internally.
	 */
	abstract protected void close(boolean isInternalCall);

	/**
	 * This will close the current connection if open. If this is an internal
	 * call then we just set the m_IsClosing flag. If it is called with a false
	 * then we set the flag and wait for the thread to complete. At that point
	 * we Clear the thread object and return.
	 * 
	 * @param isInternalCall Is this function being called internally. 
	 */
	protected void closeBase(boolean isInternalCall) {

		// Set the closing Flag
		m_IsClosing = true;

		// Wait for exit
		if (isInternalCall == false) {
			// User initiated Close
			while (m_WorkerThreadActive) {
				try {
					Thread.sleep(100);
				}
				catch (InterruptedException ignore) {
				}
			}
		}
		else {
			// internally called close
		}

		return;
	}

	/**
	 * This will read the existing data from the device creating the byte array
	 * object and returning the number of bytes in the array.
	 * 
	 * @param buffer Buffer to write data into.
	 * @return The number of bytes written into buffer.
	 * @throws IOException Exception thrown if read fails badly.
	 */
	abstract protected int innerRead(byte[] buffer) throws IOException;

	/**
	 * This will write the provided buffer to the device.
	 * 
	 * @param buffer Buffer to write data from.
	 * @throws IOException Exception thrown if write fails badly.
	 */
	abstract protected void innerWrite(byte[] buffer) throws IOException;

	/**
	 * This will return true if we have connected to a new client.
	 * 
	 * @return If we connected successfully true, otherwise false.
	 */
	abstract protected boolean innerListen();

	/**
	 * A summary of the configuration.
	 * 
	 * @return A summary of the configuration.
	 */
	abstract protected String configSummary();
	
	/**
	 * A single line description of the configuration.
	 * 
	 * @return A single line description of the configuration. 
	 */
	abstract protected String configCompact();

	/**
	 * A detailed description of the configuration.
	 * 
	 * @return A detailed description of the configuration.
	 */
	abstract protected String configDetail();

	/**
	 * A detailed description of the configuration. This is a verbose version of
	 * the connection configuration.
	 */
	@Override
	public String toString() {
		return toString("L");
	}

	/**
	 * A description of the configuration. This version allows the detail
	 * verboseness to be specified.
	 * 
	 * @param format Format of the data to be returned.  'L' is for 
	 * long, 'C' is for compact and 'S' is for short.
	 * @return This will return a string with a description of 
	 * the configuration.
	 */
	public String toString(String format) {
		String results = "";

		// Print out line by line
		if (format == "L") {
			// Long Format
			results = configDetail();
		}
		else if (format == "S") {
			// Short Format
			results = configSummary();
		}
		else {
			// Unknown
			throw new IllegalArgumentException("ToString parameter must be 'L' or 'S'");
		}

		return results;
	}

	/**
	 * This will start the communication thread and handle all of the
	 * asynchronous reading and writing to the device.
	 */
	@Override
	public void run() {
		boolean doRunRun = true;

		while (doRunRun) {
			try {
				// Wait until run is triggered
				synchronized(this) {
					while (!m_WorkerThreadActive)
						wait(1000);
				}

				// Run our thread
				runInternal();
			} catch (Exception ignore){}

			// If we have made it out of the loop the 'thread' has finished
			m_WorkerThreadActive = false;
		}

		return;
	}

	/**
	 * This is the internal version of run so the external facing version
	 * can mimic the starting and stopping of the thread.
	 */
	private void runInternal() {
		int length;
		boolean oneTry = true;
		boolean shouldSleep = false;
		boolean doReconnect = false;
		byte[] buffer = new byte[1024 * 64];
		byte[] sendItem = null;

		// Reset Stuff
		m_Reconnecting = false;
		m_IsClosing = false;

		try {
			// ----------------------------------------------------------------
			// Open Check
			// ----------------------------------------------------------------
			if (m_IsOpen == false) {
				// We arn't open
				m_IsActive = false;
				m_IsClosing = true;
			}

			// ----------------------------------------------------------------
			// Listen
			// ----------------------------------------------------------------
			while ((oneTry) || (m_IsClosing == false)) {

				// Clear one try, now you are on your own
				oneTry = false;

				// Wait until we get a connection
				if (getIsServerMode()) {
					// We are in server mode to wait until we get a connection
					if (m_IsActive == true) {
						// Set to false
						m_IsActive = false;
					}

					if (innerListen() == false) {
						// No connections waiting
						try {
							Thread.sleep(100);
						}
						catch (InterruptedException ignore) {
						}
						continue;
					}
				}

				// Get the data stream they are sending us
				m_IsActive = true;

				// ----------------------------------------------------------------
				// Data process loop
				// ----------------------------------------------------------------
				while (getIsActive()) {
					// If we performed no action then sleep
					if (shouldSleep && !doReconnect)
						try {
							Thread.sleep(100);
						}
						catch (InterruptedException ignore) {
						}
					shouldSleep = true;

					// ----------------------------------------------------------------
					// Check for reconnect to client
					// ----------------------------------------------------------------
					if (doReconnect) {
						// A reconnect was signaled
						if (Monitor.tryEnter(m_LockGeneral, 0)) {
							try {
								// First close our existing connection
								doReconnect = false;
								m_Reconnecting = true;

								// This could cause a problem. If close is
								// called externally then the lock
								// could be waiting for m_IsActive to be set to
								// false before it releases.
								// If this then comes along and waits in close
								// for the external call to
								// finish we will dead lock.
								close(true);

								// Give the close time to settle
								try {
									Thread.sleep(500);
								}
								catch (InterruptedException ignore) {
								}

								// Reopen
								open();

								// Remove the is closing flag
								m_IsClosing = false;

								// If server skip out of loop
								if (getIsServerMode()) {
									// Go back to waiting for a connection
									m_IsActive = false;
									continue;
								}
							}
							catch (Exception error) {
								// Eeek
								throw new Exception("Reconnect failed: "
										+ error.getLocalizedMessage());
							}
							finally {
								// Release the lock
								Monitor.exit(m_LockGeneral);
							}
						}
						else if (m_IsClosing == true) {
							// Close was called so don't try to reconnect
							break;
						}
						else {
							// Not in close wait loop or was open so just try
							// again
							continue;
						}
					}

					// ----------------------------------------------------------------
					// Send any existing data in the stream
					// ----------------------------------------------------------------
					synchronized (m_DataLockSend) {
						sendItem = null;
						m_ClearTriggered = false;

						if (m_DataStorageSend.size() > 0) {
							// Add to temp variable to stop blocking on slow
							// connections
							sendItem = m_DataStorageSend.get(0);
						}
					}

					if (sendItem != null) {
						// Send the first item in the list
						try {
							// We performed an action
							shouldSleep = false;

							// Send data
							innerWrite(sendItem);

							synchronized (m_DataLockSend) {
								if (m_ClearTriggered == false) {
									// Clear wasn't triggered so remove first
									// element
									m_DataStorageSend.remove(0);
								}
							}
						}
						catch (Exception ignore) {
							// Reconnect
							doReconnect = true;

							// Restart while so will reconnect
							continue;
						}
					}

					// ----------------------------------------------------------------
					// If there is nothing left to send and we are closing then
					// exit loop
					// ----------------------------------------------------------------
					if (m_IsClosing && (m_DataStorageSend.size() == 0))
						break;

					// ----------------------------------------------------------------
					// Read Data
					// ----------------------------------------------------------------
					if (getHasData()) {
						// There is data to read
						synchronized (m_DataLockRecv) {
							length = innerRead(buffer);

							// Read happened
							m_mStreamRecv.write(buffer, 0, length);

							// We performed an action
							shouldSleep = false;
						}
					}
				}
			}
		}
		catch (Exception ignore) {
		}

		// ----------------------------------------------------------------
		// Close
		// ----------------------------------------------------------------
		try {
			m_Reconnecting = false;

			// This could cause a problem. If close is called externally then
			// the lock
			// could be waiting for m_IsActive to be set to false before it
			// releases.
			// If this then comes along and waits in close for the external call
			// to
			// finish we will dead lock.
			close(true);

			// We are closed
			m_IsActive = false;
			m_IsOpen = false;
			m_IsClosing = false;
		}
		finally {
			clearWriteBuffer();
		}

		return;
	}
}
