package datamaxoneil;

import java.util.HashMap;
import java.lang.IllegalStateException;

/**
 * This class implements a C# style Monitor class. This allows us to try a lock
 * and time out if the object is locked.
 * 
 * @author Datamax-O'Neil
 * @version 2.0.1 (05 Sept 2013)
 */
public class Monitor {

	/** This internal class hold the lock data. */
	private class LockData {
		/** The thread that owns this lock */
		public Thread owningThread = null;
		
		/** How many locks have been set by the owning thread. */
		public long lockCount = 0;
	}

	/** Lock Gateway Object */
	private static Object m_LockGateway = new Object();

	/** Locks */
	private static HashMap<Object, LockData> m_Locks = new HashMap<Object, LockData>();

	/**
	 * Acquires an exclusive lock on the specified object.
	 * <p>
	 * Use Enter to acquire the Monitor on the object passed as the parameter.
	 * If another thread has executed an Enter on the object, but has not yet
	 * executed the corresponding Exit, the current thread will block until the
	 * other thread releases the object. It is legal for the same thread to
	 * invoke Enter more than once without it blocking; however, an equal number
	 * of Exit calls must be invoked before other threads waiting on the object
	 * will unblock.
	 * 
	 * @param lockObject The object on which to acquire the monitor lock.
	 * @throws IllegalArgumentException if the lockObject is null
	 * @since 1.0.0
	 */
	public static void enter(Object lockObject) {

		// Parameter Check
		if (lockObject == null) {
			throw new IllegalArgumentException("lockObject was null");
		}

		// Wait forever to get the lock
		while (tryEnter(lockObject, 0) == false) {
			// Wait until we can try again
			Thread.yield();
		}

		return;
	}

	/**
	 * Releases an exclusive lock on the specified object.
	 * <p>
	 * The calling thread must own the lock on the lockObject parameter. If the
	 * calling thread owns the lock on the specified object, and has made an
	 * equal number of Exit and Enter calls for the object, then the lock is
	 * released. If the calling thread has not invoked Exit as many times as
	 * Enter, the lock is not released.
	 * 
	 * @param lockObject The object on which to acquire the monitor lock.
	 * @throws IllegalArgumentException if the lockObject is null
	 * @throws IllegalStateException if lockObject is currently locked by
	 *         another thread.
	 * @throws IllegalStateException if lockObject is not currently locked.
	 * @since 1.0.0
	 */
	public static void exit(Object lockObject) {

		// Parameter Check
		if (lockObject == null) {
			throw new IllegalArgumentException("lockObject was null");
		}

		synchronized (m_LockGateway) {
			// See if it is currently locked
			if (!m_Locks.containsKey(lockObject)) {
				// DNE
				throw new IllegalStateException("Monitor.exit called on object that is not currently locked.");
			}

			// Get Lock Data
			LockData lData = m_Locks.get(lockObject);

			// Check that it is actually our lock
			if (lData.owningThread != Thread.currentThread()) {
				// Incorrect Thread
				throw new IllegalStateException("Monitor.exit called on object that is lock by another thread.");
			}

			// Our lock so decrement the count
			lData.lockCount--;

			// If no more locks then remove the object
			if (lData.lockCount == 0) {
				// Remove
				m_Locks.remove(lockObject);
			}
		}

		return;
	}

	/**
	 * Attempts, for the specified number of milliseconds, to acquire an
	 * exclusive lock on the specified object.
	 * 
	 * @param lockObject The object on which to acquire the monitor lock.
	 * @param timeout The number of milliseconds to wait for the lock.
	 * @return true if the current thread acquires the lock; otherwise, false.
	 * @throws IllegalArgumentException if the lockObject is null
	 * @throws IllegalArgumentException if the timeout is negative
	 * @since 1.0.0
	 */
	public static boolean tryEnter(Object lockObject, long timeout) {
		boolean result = false;

		// Parameter Check
		if (lockObject == null) {
			throw new IllegalArgumentException("lockObject was null");
		}

		if (timeout < 0) {
			throw new IllegalArgumentException("timeout was negative");
		}

		// Try to acquire the lock for the given time period. This needs
		// to be designed so the locking of m_LockGateway doesn't block
		// multiple calls into this function.
		long startTime = System.currentTimeMillis();

		do {
			// Attempt Quickly
			synchronized (m_LockGateway) {
				// Try to claim
				if (!m_Locks.containsKey(lockObject)) {
					// DNE so claim
					LockData lData = (new Monitor()).new LockData();
					lData.owningThread = Thread.currentThread();
					lData.lockCount = 1;

					// Add
					m_Locks.put(lockObject, lData);
					result = true;
				}
				else {
					// See if is our lock
					LockData lData = m_Locks.get(lockObject);

					// Check that it is actually our lock
					if (lData.owningThread == Thread.currentThread()) {
						// Its ours so increment count
						lData.lockCount += 1;
						result = true;
					}
				}
			}

			// Sleep a bit
			Thread.yield();
		}
		while (((System.currentTimeMillis() - startTime) <= timeout) && (result == false));

		return result;
	}

}
