/*
 * Decompiled with CFR 0.152.
 */
package com.anahata.util.io;

import com.anahata.util.io.CompressionType;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.InflaterInputStream;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.compress.compressors.CompressorException;
import org.apache.commons.compress.compressors.CompressorStreamFactory;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class AnahataCompressionUtils {
    private static final Logger log = LoggerFactory.getLogger(AnahataCompressionUtils.class);
    public static final String HEADER = "AnahataCompression";
    private static final CompressorStreamFactory compressorStreamFactory = new CompressorStreamFactory();

    public static CompressionType readHeader(HttpServletRequest request) {
        return AnahataCompressionUtils.parse(request.getHeader(HEADER));
    }

    public static CompressionType readHeader(Message message) throws JMSException {
        return AnahataCompressionUtils.parse(message.getStringProperty(HEADER));
    }

    public static CompressionType parse(String serialization) {
        CompressionType ret = null;
        if (serialization != null) {
            ret = CompressionType.valueOf(serialization);
        }
        return ret;
    }

    public static String getRatio(long uncompressed, long compressed) {
        double ratio = 100L * (compressed + 1L) / (uncompressed + 1L);
        return (int)(100.0 - ratio) + "%";
    }

    public static File compress(CompressionType compressionType, File source) throws IOException, CompressorException {
        long ts = System.currentTimeMillis();
        log.debug("Compressing {}", (Object)source);
        File tempFile = File.createTempFile("anahata-compress-", "." + compressionType.toString().toLowerCase());
        tempFile.deleteOnExit();
        FileOutputStream fos = new FileOutputStream(tempFile);
        try (OutputStream compressorOutputStream = AnahataCompressionUtils.newCompressingOutputStream(compressionType, fos);
             FileInputStream fis = new FileInputStream(source);){
            IOUtils.copy((InputStream)fis, (OutputStream)compressorOutputStream);
        }
        ts = System.currentTimeMillis() - ts;
        log.debug("Compressing {} took {} ms, original {}, compressed {}, ratio {}", new Object[]{source, ts, source.length(), tempFile.length(), AnahataCompressionUtils.getRatio(source.length(), tempFile.length())});
        return tempFile;
    }

    public static byte[] compress(CompressionType compressionType, byte[] source) throws IOException, CompressorException {
        if (compressionType == CompressionType.NONE) {
            return source;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        OutputStream os = AnahataCompressionUtils.newCompressingOutputStream(compressionType, baos);
        IOUtils.write((byte[])source, (OutputStream)os);
        os.flush();
        os.close();
        byte[] ret = baos.toByteArray();
        if (log.isDebugEnabled()) {
            BigDecimal ratio = new BigDecimal(100.0f * ((float)ret.length / (float)source.length));
            ratio = ratio.setScale(2, RoundingMode.HALF_UP);
            log.trace("{} compression: source={}, compressed={}, ratio={}%", new Object[]{compressionType, source.length, ret.length, ratio});
            if (ratio.intValue() > 100) {
                log.trace("{} compression made data bigger: source={}, compressed={}, ratio={}%", new Object[]{compressionType, source.length, ret.length, ratio});
            }
        }
        return ret;
    }

    public static byte[] uncompress(CompressionType compressionType, byte[] source) throws IOException, CompressorException {
        if (compressionType == CompressionType.NONE) {
            return source;
        }
        ByteArrayInputStream bais = new ByteArrayInputStream(source);
        InputStream is = AnahataCompressionUtils.newUncompressingInputStream(compressionType, bais);
        byte[] ret = IOUtils.toByteArray((InputStream)is);
        log.trace("{} decompressed size : {}", (Object)compressionType, (Object)ret.length);
        return ret;
    }

    public static InputStream newUncompressingInputStream(CompressionType compressionType, InputStream is) throws CompressorException {
        if (CompressionType.NONE != compressionType) {
            is = CompressionType.DEFLATE == compressionType ? new InflaterInputStream(is) : compressorStreamFactory.createCompressorInputStream(compressionType.name(), is);
        }
        log.trace("InputStream: {}", (Object)is.getClass().getSimpleName());
        return is;
    }

    public static OutputStream newCompressingOutputStream(CompressionType compressionType, OutputStream os) throws CompressorException {
        if (CompressionType.NONE != compressionType) {
            os = CompressionType.DEFLATE == compressionType ? new DeflaterOutputStream(os, new Deflater(9)) : compressorStreamFactory.createCompressorOutputStream(compressionType.name(), os);
        }
        return os;
    }

    private AnahataCompressionUtils() {
    }
}

